-- =========================
-- Brands
-- =========================
CREATE TABLE wood_brands (
  id                   INTEGER PRIMARY KEY AUTOINCREMENT,
  brand_name           TEXT NOT NULL,
  brand_img            TEXT,
  brand_folder         TEXT NOT NULL,
  "enable"             INTEGER NOT NULL DEFAULT 1,
  "deleted"            INTEGER NOT NULL DEFAULT 0,
  "order"              INTEGER,
  view_product_info    TEXT NOT NULL DEFAULT '',
  view_product_details TEXT NOT NULL DEFAULT '',
  created              TEXT DEFAULT CURRENT_TIMESTAMP,
  last_updated         TEXT DEFAULT CURRENT_TIMESTAMP
);

CREATE TRIGGER trg_brands_last_updated
BEFORE UPDATE ON wood_brands
FOR EACH ROW
BEGIN
  UPDATE wood_brands SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

-- =========================
-- Product
-- =========================
CREATE TABLE wood_product (
  id               INTEGER PRIMARY KEY AUTOINCREMENT,
  brand_id         INTEGER,
  "no"             TEXT NOT NULL,
  decor_code       TEXT,
  decor_name       TEXT,
  decor_group      TEXT NOT NULL,
  core             TEXT,
  is_new           INTEGER DEFAULT 0,
  is_not_available INTEGER DEFAULT 0,
  deleted          INTEGER NOT NULL DEFAULT 0,
  created          TEXT DEFAULT CURRENT_TIMESTAMP,
  last_updated     TEXT DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (brand_id) REFERENCES wood_brands(id)
);

CREATE TRIGGER trg_product_last_updated
BEFORE UPDATE ON wood_product
FOR EACH ROW
BEGIN
  UPDATE wood_product SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

-- Helpful indexes
CREATE INDEX idx_product_brand_id ON wood_product(brand_id);

-- =========================
-- Product Dimensions
-- =========================
CREATE TABLE wood_product_dimensions (
    id             INTEGER PRIMARY KEY AUTOINCREMENT,
    product_id     INTEGER,
    width          INTEGER,
    "length"       INTEGER,
    deleted        INTEGER DEFAULT 0,
    last_updated   TEXT DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES wood_product(id)
);

CREATE TRIGGER trg_product_dimensions_last_updated
BEFORE UPDATE ON wood_product_dimensions
FOR EACH ROW
BEGIN
  UPDATE wood_product_dimensions SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

CREATE INDEX idx_product_dimensions_product_id ON wood_product_dimensions(product_id);

-- =========================
-- Edge Banding
-- =========================
CREATE TABLE wood_edge_banding (
    id              INTEGER PRIMARY KEY AUTOINCREMENT,
    brand_id        INTEGER,
    board_thickness INTEGER NOT NULL,
    edge_thickness  TEXT NOT NULL,
    banding_details TEXT NOT NULL,
    banding_type    TEXT NOT NULL,
    deleted         INTEGER DEFAULT 0,
    last_updated    TEXT DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (brand_id) REFERENCES wood_brands(id)
);

CREATE TRIGGER trg_edge_banding_last_updated
BEFORE UPDATE ON wood_edge_banding
FOR EACH ROW
BEGIN
  UPDATE wood_edge_banding SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

CREATE INDEX idx_edge_banding_brand_id ON wood_edge_banding(brand_id);

-- =========================
-- Edge Thicknesses Prices
-- =========================
CREATE TABLE wood_edge_thicknesses_prices (
    id               INTEGER PRIMARY KEY AUTOINCREMENT,
    edge_banding_id  INTEGER NOT NULL,
    thickness_range  TEXT NOT NULL,
    price_per_meter  REAL NOT NULL,
    last_updated     TEXT DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (edge_banding_id) REFERENCES wood_edge_banding(id)
);

CREATE TRIGGER trg_edge_thicknesses_prices_last_updated
BEFORE UPDATE ON wood_edge_thicknesses_prices
FOR EACH ROW
BEGIN
  UPDATE wood_edge_thicknesses_prices SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

CREATE INDEX idx_edge_thicknesses_prices_banding_id ON wood_edge_thicknesses_prices(edge_banding_id);

-- =========================
-- Product History
-- =========================
CREATE TABLE wood_product_history (
    history_id       INTEGER PRIMARY KEY AUTOINCREMENT,
    product_id       INTEGER,
    brand_id         INTEGER,
    user_id          INTEGER,
    no               TEXT,
    decore_code      TEXT,
    decore_group     TEXT,
    decore_name      TEXT,
    core             TEXT,
    is_new           INTEGER DEFAULT 0,
    is_not_available INTEGER DEFAULT 1,
    dimensions_price TEXT,
    images           TEXT,
    deleted          INTEGER DEFAULT 0,
    change_type      TEXT CHECK (change_type IN ('INSERT','UPDATE','DELETE')),
    change_timestamp TEXT DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES wood_product(id),
    FOREIGN KEY (brand_id)  REFERENCES wood_brands(id)
);

CREATE INDEX idx_product_history_product_id ON wood_product_history(product_id);
CREATE INDEX idx_product_history_brand_id   ON wood_product_history(brand_id);

-- =========================
-- Product Pricing
-- =========================
CREATE TABLE wood_product_pricing (
    id            INTEGER PRIMARY KEY AUTOINCREMENT,
    dimensions_id INTEGER,
    thickness     REAL,
    price         REAL,
    deleted       INTEGER DEFAULT 0,
    last_updated  TEXT DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (dimensions_id) REFERENCES wood_product_dimensions(id)
);

CREATE TRIGGER trg_product_pricing_last_updated
BEFORE UPDATE ON wood_product_pricing
FOR EACH ROW
BEGIN
  UPDATE wood_product_pricing SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

CREATE INDEX idx_product_pricing_dimensions_id  ON wood_product_pricing(dimensions_id);

-- =========================
-- Product Images
-- =========================
CREATE TABLE wood_product_images (
    id             INTEGER PRIMARY KEY AUTOINCREMENT,
    product_id     INTEGER,
    "no"           INTEGER,
    image_path     TEXT,
    image_location TEXT,
    deleted        INTEGER DEFAULT 0,
    last_updated   TEXT DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES wood_product(id)
);

CREATE TRIGGER trg_product_images_last_updated
BEFORE UPDATE ON wood_product_images
FOR EACH ROW
BEGIN
  UPDATE wood_product_images SET last_updated = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

CREATE INDEX idx_product_images_product_id ON wood_product_images(product_id);
