CREATE TABLE tenant (
    id                      INT AUTO_INCREMENT PRIMARY KEY,
    parent_id               INT DEFAULT NULL,
    company_name            VARCHAR(255) NOT NULL,
    domain                  VARCHAR(255) NOT NULL,
    company_number          VARCHAR(255) NOT NULL,
    payment_provider        ENUM('stripe', 'paypal', 'square', 'braintree') DEFAULT 'stripe',
    payment_api_key         VARCHAR(255) DEFAULT NULL,
    minimum_order_amount    DECIMAL(10, 2) DEFAULT 0.00,
    small_order_fee         DECIMAL(10, 2) DEFAULT 0.00,
    food_hygiene_id         VARCHAR(255) NULL,
    primary_contact_name    VARCHAR(255) NOT NULL,
    tenant_status           ENUM('active', 'pending', 'suspended', 'deleted', 'hidden') DEFAULT 'pending',
    max_distance            DECIMAL(10, 2) DEFAULT 0.00,
    active_delivery         TINYINT(1) DEFAULT 1,
    active_pickup           TINYINT(1) DEFAULT 1,
    is_deleted              TINYINT(1) DEFAULT 0,
    created_at              DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at              DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

--INSERT INTO `tenant` (`id`, `parent_id`, `company_name`, `domain`, `company_number`, `food_hygiene_id`, `primary_contact_name`, `tenant_status`, `is_deleted`, `created_at`, `updated_at`) 
--VALUES (NULL, NULL, 'Waitron.uk', '', '', NULL, '', 'pending', '0', current_timestamp(), current_timestamp());

CREATE TABLE tenant_contact (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id       INT NOT NULL,
    user_id         INT DEFAULT NULL,  -- Optional link to user (i.e. owner)
    contact_type    ENUM('mobile', 'email', 'address', 'business', 'phone', 'name') NOT NULL,
    is_primary      ENUM('yes', 'no') NOT NULL DEFAULT 'no',
    label           VARCHAR(255) NOT NULL,
    contact_value   VARCHAR(255) NOT NULL,
    is_deleted      TINYINT(1) DEFAULT 0,
    created_at      DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at      DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE SET NULL
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;


CREATE TABLE tenant_theme (
    id                      INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id               INT NOT NULL,
    theme_name              VARCHAR(100) NOT NULL,
    primary_color           VARCHAR(7) NOT NULL,
    secondary_color         VARCHAR(7) NOT NULL,
    text_color              VARCHAR(7) NOT NULL,
    secondary_text_color    VARCHAR(7) NOT NULL,
    accent_color            VARCHAR(7) NOT NULL,
    background_color        VARCHAR(7) NOT NULL,
    button_color            VARCHAR(7) NOT NULL,
    theme_status            ENUM('active', 'draft', 'archived') DEFAULT 'draft',
    is_applied              ENUM('yes', 'no') DEFAULT 'no',
    is_deleted              TINYINT(1) DEFAULT 0,
    created_at              DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at              DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_theme_name_per_tenant (tenant_id, theme_name),
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

CREATE TABLE tenant_business_hours (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id       INT NOT NULL,
    hours_category  ENUM('store', 'delivery', 'pickup', 'special') NOT NULL,   -- replaces `type`
    opening_day     ENUM('Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday') DEFAULT NULL, -- replaces `day`
    specific_date   DATE DEFAULT NULL,                    -- replaces `date` (used for special hours)
    open_time       TIME DEFAULT NULL,                        -- replaces `start`
    close_time      TIME DEFAULT NULL,                       -- replaces `end`
    is_closed       TINYINT(1) DEFAULT 0 NOT NULL,            -- 1 = closed, 0 = open
    is_deleted      TINYINT(1) DEFAULT 0,                    -- soft delete flag
    created_at      DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at      DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

CREATE TABLE tenant_history (
    id              INT PRIMARY KEY AUTO_INCREMENT,
    tenant_id       INTEGER NOT NULL,
    action_type     ENUM('updated', 'deleted', 'expired') NOT NULL,
    table_of_source VARCHAR(50) NOT NULL,  -- tenant, tenant_contact, etc.
    old_value       TEXT DEFAULT NULL,    -- JSON string of old values
    action_by       INT NOT NULL,        -- user_id or system
    action_at       DATETIME DEFAULT CURRENT_TIMESTAMP
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;


CREATE TABLE tenant_delivery_rules (
    id                      INTEGER PRIMARY KEY AUTO_INCREMENT,
    tenant_id               INTEGER NOT NULL,
    minimum_delivery_amount DECIMAL(10, 2) NOT NULL,
    delivery_charge         DECIMAL(10, 2) NOT NULL,
    is_active               BOOLEAN NOT NULL DEFAULT 0,
    is_deleted              BOOLEAN NOT NULL DEFAULT 0,
    created_at              DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at              DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

CREATE TABLE tenant_admin_fees (
    id                  INTEGER PRIMARY KEY AUTO_INCREMENT,
    tenant_id           INTEGER NOT NULL,
    customer_admin_fee  REAL NOT NULL DEFAULT 0.00,  -- what the customer sees/pays
    tenant_admin_fee    REAL NOT NULL DEFAULT 0.00,  -- what you charge the tenant privately
    is_active           INTEGER NOT NULL DEFAULT 0,  -- 1 = active, 0 = inactive
    is_deleted          INTEGER NOT NULL DEFAULT 0,  -- soft delete flag
    created_at          DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at          DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- 1) Active table: 1 active pause row per tenant
CREATE TABLE IF NOT EXISTS tenant_pause_active (
    tenant_id             INT NOT NULL,
    pause_type            ENUM('UNTIL_CLOSED','MINUTES_15','HOURS_1','HOURS_2','CUSTOM') NOT NULL,
    is_closed_collection  BOOLEAN NOT NULL DEFAULT FALSE,
    is_closed_delivery    BOOLEAN NOT NULL DEFAULT FALSE,
    pause_start_at        DATETIME NOT NULL,
    pause_until           DATETIME NOT NULL,
    set_by_user_id        INT NULL,
    created_at            DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at            DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (tenant_id),
    INDEX idx_tenant_pause_active_pause_until (pause_until),
    INDEX idx_tenant_pause_active_tenant_until (tenant_id, pause_until)
) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;


ALTER TABLE tenant_business_hours
  ADD CONSTRAINT chk_hours_date_or_day
  CHECK (
    (specific_date IS NOT NULL AND opening_day IS NULL)
    OR (specific_date IS NULL AND opening_day IS NOT NULL)
  );

ALTER TABLE tenant_history
  ADD CONSTRAINT fk_action_by_user FOREIGN KEY (action_by) REFERENCES user_login(pk_user_id);
  
CREATE OR REPLACE VIEW tenant_upcoming_hours AS
    SELECT
        d.generated_date,
        bh.tenant_id,
        bh.hours_category,
        bh.opening_day,
        CASE WHEN bh.is_closed = 1 THEN NULL ELSE bh.open_time END AS open_time,
        CASE WHEN bh.is_closed = 1 THEN NULL ELSE bh.close_time END AS close_time,
        bh.is_closed
    FROM (
        -- Generate the next 14 days
        SELECT CURDATE() + INTERVAL seq DAY AS generated_date,
            DAYNAME(CURDATE() + INTERVAL seq DAY) AS day_name
        FROM (
            SELECT 0 AS seq UNION ALL SELECT 1 UNION ALL SELECT 2 UNION ALL SELECT 3 UNION ALL SELECT 4
            UNION ALL SELECT 5 UNION ALL SELECT 6 UNION ALL SELECT 7 UNION ALL SELECT 8 UNION ALL SELECT 9
            UNION ALL SELECT 10 UNION ALL SELECT 11 UNION ALL SELECT 12 UNION ALL SELECT 13
        ) AS days
    ) AS d
    JOIN tenant_business_hours bh ON (
        (
            -- Special hours: take precedence
            bh.specific_date = d.generated_date
            AND bh.is_deleted = 0
        )
        OR (
            -- Regular hours: only used if NO special for that tenant on that date
            bh.specific_date IS NULL
            AND bh.opening_day = d.day_name
            AND bh.is_deleted = 0
            AND NOT EXISTS (
                SELECT 1 FROM tenant_business_hours sp
                WHERE 
                    sp.tenant_id = bh.tenant_id
                    AND sp.specific_date = d.generated_date
                    AND sp.is_deleted = 0
            )
        )
    )
    WHERE bh.is_deleted = 0
    and bh.is_closed = 0
    ORDER BY bh.tenant_id, d.generated_date, bh.hours_category;


INSERT INTO `tenant_theme` (`id`, `tenant_id`, `theme_name`, `primary_color`, `secondary_color`, `text_color`, `secondary_text_color`, `accent_color`, `background_color`, `button_color`, `theme_status`, `is_applied`, `is_deleted`, `created_at`, `updated_at`) VALUES (0, '', 'Default', '#FFFFFF', '#F0F0F0', '#333333', '#777777', '#007BFF', '#FAFAFA', '#007BFF', 'draft', 'yes', '0', current_timestamp(), current_timestamp());

-- Trigger to automatically set open_time and close_time to NULL when is_closed is set to 1
DELIMITER //
CREATE TRIGGER tr_business_hours_auto_null_times
    BEFORE UPDATE ON tenant_business_hours
    FOR EACH ROW
BEGIN
    IF NEW.is_closed = 1 THEN
        SET NEW.open_time = NULL;
        SET NEW.close_time = NULL;
    END IF;
END//
DELIMITER ;