<?php
function insertNewTenant($inputData) {
    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    // 🔍 Collect validation errors
    $errors = [];

    if (empty($inputData['tenant']['companyName'])) {
        $errors[] = 'Company name is required';
    }

    if (empty($inputData['tenant']['domain'])) {
        $errors[] = 'Domain is required';
    }

    if (empty($inputData['tenant']['companyNumber'])) {
        $errors[] = 'Company number is required';
    }

    if (empty($inputData['tenant']['primaryContactName'])) {
        $errors[] = 'Primary contact name is required';
    }

    if (empty($inputData['tenant']['tenantStatus'])) {
    } else {
        $validStatuses = ['pending', 'active', 'suspended', 'deleted', 'hidden'];
        if (!in_array($inputData['tenant']['tenantStatus'], $validStatuses)) {
            $errors[] = 'Invalid tenant status';
        }
    }

    if(empty($inputData['tenant']['minimumOrderAmount']) || !is_numeric($inputData['tenant']['minimumOrderAmount'])) {
        $errors[] = 'Minimum order amount must be a number';
    }

    if(empty($inputData['tenant']['smallOrderFee']) || !is_numeric($inputData['tenant']['smallOrderFee'])) {
        $errors[] = 'Small order fee must be a number';
    }

    // flatten errors array with , separator
    $errors = implode(', ', $errors);

    // ❌ Return all validation errors at once
    if (!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => $errors
        ];
    }


    $domain = 'https://'.$inputData['tenant']['domain'];

    // ✅ Proceed with insert
    $sql = "INSERT INTO tenant  (company_name, domain, company_number, food_hygiene_id, primary_contact_name, tenant_status, payment_provider, payment_api_key, 
                                  minimum_order_amount, small_order_fee) 
            VALUES              (:companyName, :domain, :companyNumber, :foodHygieneId, :primaryContactName, :tenantStatus, :paymentProvider, :paymentApiKey, 
                                  :minimumOrderAmount, :smallOrderFee)";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':companyName',        $inputData['tenant']['companyName']);
    $stmt->bindParam(':domain',             $domain);
    $stmt->bindParam(':companyNumber',      $inputData['tenant']['companyNumber']);
    $stmt->bindParam(':foodHygieneId',      $inputData['tenant']['foodHygieneId']);
    $stmt->bindParam(':primaryContactName', $inputData['tenant']['primaryContactName']);
    $stmt->bindParam(':paymentProvider',    $inputData['tenant']['paymentProvider']);
    $stmt->bindParam(':paymentApiKey',      $inputData['tenant']['paymentApiKey']);
    $stmt->bindParam(':minimumOrderAmount', $inputData['tenant']['minimumOrderAmount']);
    $stmt->bindParam(':smallOrderFee',      $inputData['tenant']['smallOrderFee']);
    if(isset($inputData['tenant']['tenantStatus'])) {
        $stmt->bindParam(':tenantStatus', $inputData['tenant']['tenantStatus']);
    } else {
        $stmt->bindValue(':tenantStatus', 'pending');
    }
    $stmt->execute();

    
    $lastInsertId = $pdo->lastInsertId();
    return [
        'status' => 'success',
        'message' => 'Tenant inserted successfully',
        'tenantId' => $lastInsertId
    ];
}
function insertHistoryRecord($inputData){

    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $sql = "INSERT INTO tenant_history  (action_type, table_of_source,  old_value,  action_by, tenant_id) 
            VALUES                      (:actionType, :tableOfSource,   :oldData,   :userId, :tenantId)"; 
    $stmt = $pdo->prepare($sql);
 

    foreach ($inputData['his'] as $rowJson) {
        if (isset($inputData['tenantId'])) {
            $tenantId = $inputData['tenantId'];
        } else {
            $tenantId = 0;
        }

        $stmt->bindValue(':actionType',    $inputData['actionType']);
        $stmt->bindValue(':tableOfSource', $inputData['tableName']);
        $stmt->bindValue(':oldData',       $rowJson);
        $stmt->bindValue(':userId',        $inputData['userId']);
        $stmt->bindValue(':tenantId',      $tenantId); // hardcoded for now, should be passed in the inputData array
        $stmt->execute();

        $lastInsertId[] = $pdo->lastInsertId(); // overwritten each time
    }

    return [
        'status' => 'success',
        'message' => 'History record inserted successfully',
        'historyId' => $lastInsertId
    ];



}
function insertNewTenantTheme($inputData){
    $errors = [];

    $tableColumns = tableColumnMapping();

    // make sure that we have all the columns in the tenantTheme table
    foreach ($tableColumns['tenantTheme'] as $key => $config) {
        $isOptional = isset($config['optional']) && $config['optional'] == 1;
        
        if (!$isOptional && !array_key_exists($key, $inputData['tenantTheme'])) {
            $errors[] = "$key is missing";
        }
    }
    

    foreach ($inputData['tenantTheme'] as $key => $value) {
        if (empty($value)) {
            $errors[] = "$key is required";
        }
    }

    if(!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => implode(', ', $errors)
        ];
    }

    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $sql = "INSERT INTO tenant_theme (tenant_id, theme_name, primary_color, secondary_color, text_color, secondary_text_color, accent_color, background_color, button_color) 
            VALUES (:tenantId, :themeName, :primaryColor, :secondaryColor, :textColor, :secondaryTextColor, :accentColor, :backgroundColor, :buttonColor)";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':tenantId',           $inputData['tenantId']);
    $stmt->bindParam(':themeName',          $inputData['tenantTheme']['themeName']);
    $stmt->bindParam(':primaryColor',       $inputData['tenantTheme']['primaryColor']);
    $stmt->bindParam(':secondaryColor',     $inputData['tenantTheme']['secondaryColor']);
    $stmt->bindParam(':textColor',          $inputData['tenantTheme']['textColor']);
    $stmt->bindParam(':secondaryTextColor', $inputData['tenantTheme']['secondaryTextColor']);
    $stmt->bindParam(':accentColor',        $inputData['tenantTheme']['accentColor']);
    $stmt->bindParam(':backgroundColor',    $inputData['tenantTheme']['backgroundColor']);
    $stmt->bindParam(':buttonColor',        $inputData['tenantTheme']['buttonColor']);
    $stmt->execute();
    $lastInsertId = $pdo->lastInsertId();
    return [
        'status' => 'success',
        'message' => 'Tenant theme inserted successfully',
        'themeId' => $lastInsertId
    ];
}
function insertNewTenantContactAdmin($inputData){
    $errors = [];

    $tableColumns = tableColumnMapping();

    // make sure that we have all the columns in the tenantContactAdmin table
    foreach ($tableColumns['tenantContact'] as $key => $config) {
        $isOptional = isset($config['optional']) && $config['optional'] == 1;
        
        if (!$isOptional && !array_key_exists($key, $inputData['tenantContact'])) {
            $errors[] = "$key is missing";
        }
    }

    // for this we also need to check that userId is required
    if (!isset($inputData['tenantContact']['userId']) || empty($inputData['tenantContact']['userId'])) {
        $errors[] = 'userId is required';
    }
    

    foreach ($inputData['tenantContact'] as $key => $value) {
        if (empty($value)) {
            $errors[] = "$key is required";
        }
    }

    if(!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => implode(', ', $errors)
        ];
    }

    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $sql = "INSERT INTO tenant_contact (tenant_id, user_id, contact_type, is_primary, label, contact_value) 
            VALUES (:tenantId, :userId, :contactType, :isPrimary, :label, :contactValue)";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':tenantId',     $inputData['tenantId']);
    $stmt->bindParam(':userId',       $inputData['tenantContact']['userId']);
    $stmt->bindParam(':contactType',  $inputData['tenantContact']['contactType']);
    $stmt->bindParam(':isPrimary',    $inputData['tenantContact']['isPrimary']);
    $stmt->bindParam(':label',        $inputData['tenantContact']['label']);
    $stmt->bindParam(':contactValue', $inputData['tenantContact']['contactValue']);
    $stmt->execute();
    
    return [
        'status' => 'success',
        'message' => 'Tenant contact admin inserted successfully',
        'contactId' => $pdo->lastInsertId()
    ];
}
function insertNewTenantContact($inputData){
    $errors = [];

    $tableColumns = tableColumnMapping();

    // make sure that we have all the columns in the tenantContactAdmin table
    foreach ($tableColumns['tenantContact'] as $key => $config) {
        $isOptional = isset($config['optional']) && $config['optional'] == 1;
        
        if (!$isOptional && !array_key_exists($key, $inputData['tenantContact'])) {
            $errors[] = "$key is missing";
        }
    }    

    foreach ($inputData['tenantContact'] as $key => $value) {
        if (empty($value)) {
            $errors[] = "$key is required";
        }
    }

    if(!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => implode(', ', $errors)
        ];
    }

    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $sql = "INSERT INTO tenant_contact  (tenant_id,     contact_type,   is_primary,     label,      contact_value) 
            VALUES                      (:tenantId,     :contactType,   :isPrimary,     :label,     :contactValue)";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':tenantId',     $inputData['tenantId']);
    $stmt->bindParam(':contactType',  $inputData['tenantContact']['contactType']);
    $stmt->bindParam(':isPrimary',    $inputData['tenantContact']['isPrimary']);
    $stmt->bindParam(':label',        $inputData['tenantContact']['label']);
    $stmt->bindParam(':contactValue', $inputData['tenantContact']['contactValue']);
    $stmt->execute();
    
    return [
        'status' => 'success',
        'message' => 'Tenant contact admin inserted successfully',
        'contactId' => $pdo->lastInsertId()
    ];
}
function insertNewBusinessHours($inputData) {
    $errors = [];
    $insertedIds = [];

    $tableColumns = tableColumnMapping();
    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $sql = "INSERT INTO tenant_business_hours (
                tenant_id, hours_category, opening_day, specific_date,
                open_time, close_time, is_closed
            ) VALUES (
                :tenantId, :hoursCategory, :openingDay, :specificDate,
                :openTime, :closeTime, :isClosed
            )";

    $stmt = $pdo->prepare($sql);

    // Normalize input to array of rows
    $rows = isset($inputData['tenantBusinessHours'][0]) 
        ? $inputData['tenantBusinessHours']   // bulk input
        : [ $inputData['tenantBusinessHours'] ]; // single input

    foreach ($rows as $index => $data) {
        $rowErrors = [];

        foreach ($tableColumns['tenantBusinessHours'] as $key => $config) {
            $isOptional = isset($config['optional']) && $config['optional'] == 1;
            $valueExists = array_key_exists($key, $data);
            $value = $valueExists ? $data[$key] : null;

            if (!$isOptional && (!$valueExists || $value === null || $value === '')) {
                $rowErrors[] = (count($rows) > 1 ? "Row $index: " : "") . "$key is required";
            }
        }

        if (!empty($rowErrors)) {
            $errors = array_merge($errors, $rowErrors);
            continue;
        }

        // Bind and insert
        $stmt->bindParam(':tenantId',      $inputData['tenantId']);
        $stmt->bindParam(':hoursCategory', $data['hoursCategory']);
        $stmt->bindParam(':openingDay',    $data['openingDay']);
        $stmt->bindParam(':specificDate',  $data['specificDate']);
        $stmt->bindParam(':openTime',      $data['openTime']);
        $stmt->bindParam(':closeTime',     $data['closeTime']);
        $stmt->bindParam(':isClosed',      $data['isClosed']);

        $stmt->execute();
        $insertedIds[] = $pdo->lastInsertId();
    }

    if (!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => implode(', ', $errors)
        ];
    }

    return [
        'status' => 'success',
        'message' => count($insertedIds) > 1 
            ? 'All tenant business hours inserted successfully' 
            : 'Tenant business hours inserted successfully',
        'businessHoursIds' => $insertedIds
    ];
}
function insertNewTenantDeliveryRule($inputData) {
    $errors = [];
    $insertedIds = [];

    $tableColumns = tableColumnMapping();
    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $sql = "INSERT INTO tenant_delivery_rules (
                tenant_id, minimum_order_amount, delivery_charge, is_active
            ) VALUES (
                :tenantId, :minimumOrderAmount, :deliveryCharge, :isActive
            )";

    $stmt = $pdo->prepare($sql);

    // Normalize input to array of rows
    $rows = isset($inputData['tenantDeliveryRules'][0]) 
        ? $inputData['tenantDeliveryRules']   // bulk input
        : [ $inputData['tenantDeliveryRules'] ]; // single input

    foreach ($rows as $index => $data) {
        $rowErrors = [];

        foreach ($tableColumns['tenantDeliveryRules'] as $key => $config) {
            $isOptional = isset($config['optional']) && $config['optional'] == 1;
            $valueExists = array_key_exists($key, $data);
            $value = $valueExists ? $data[$key] : null;

            if (!$isOptional && (!$valueExists || $value === null || $value === '')) {
                $rowErrors[] = (count($rows) > 1 ? "Row $index: " : "") . "$key is required";
            }
        }
        // minimumOrderAmount deliveryCharge
        if (isset($data['minimumOrderAmount']) && !is_numeric($data['minimumOrderAmount'])) {
            $rowErrors[] = (count($rows) > 1 ? "Row $index: " : "") . "minimumOrderAmount must be a number";
        }
        if (isset($data['deliveryCharge']) && !is_numeric($data['deliveryCharge'])) {
            $rowErrors[] = (count($rows) > 1 ? "Row $index: " : "") . "deliveryCharge must be a number";
        }

        if (!empty($rowErrors)) {
            $errors = array_merge($errors, $rowErrors);
            continue;
        }

        // Bind and insert
        $stmt->bindParam(':tenantId',             $inputData['tenantId']);
        $stmt->bindParam(':minimumOrderAmount',   $data['minimumOrderAmount']);
        $stmt->bindParam(':deliveryCharge',       $data['deliveryCharge']);
        if (isset($data['isActive'])) {
            $stmt->bindParam(':isActive',         $data['isActive']);
        } else {
            // Default to active if not provided
            $stmt->bindValue(':isActive', 0);
        }

        try {
            $stmt->execute();
            $insertedIds[] = $pdo->lastInsertId();
        } catch (PDOException $e) {
            return  [
                'status' => 'failed',
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }

    if (!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => implode(', ', $errors)
        ];
    }

    return [
        'status' => 'success',
        'message' => count($insertedIds) > 1 
            ? 'All tenant delivery rules inserted successfully' 
            : 'Tenant delivery rule inserted successfully',
        'deliveryRuleIds' => $insertedIds
    ];
}
function insertNewTenantFees($inputData){
    $errors = [];
    $tableColumns = tableColumnMapping();

    $pdo = $inputData['db']['dbApp'];
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    foreach ($tableColumns['tenantAdminFees'] as $key => $config) {
        $isOptional = isset($config['optional']) && $config['optional'] == 1;
        
        if (!$isOptional && !array_key_exists($key, $inputData['tenantAdminFees'])) {
            $errors[] = "$key is missing";
        }
    }

    if (!empty($errors)) {
        return [
            'status' => 'failed',
            'message' => implode(', ', $errors)
        ];
    }
    // update the db is_active to 0 for the tenant if is_active is set to 1 in the new record
    if (isset($inputData['tenantAdminFees']['isActive']) && $inputData['tenantAdminFees']['isActive'] == 1) {
        $updateSql = "UPDATE tenant_admin_fees SET is_active = 0 WHERE tenant_id = :tenantId";
        $updateStmt = $pdo->prepare($updateSql);
        $updateStmt->bindParam(':tenantId', $inputData['tenantId']);
        $updateStmt->execute();
    }

    // Prepare insert statement
    $stmt = $pdo->prepare("
        INSERT INTO tenant_admin_fees (tenant_id, customer_admin_fee, tenant_admin_fee, is_active)
        VALUES (:tenantId, :customerAdminFee, :tenantAdminFee, :isActive)
    ");

    
        $stmt->bindParam(':tenantId', $inputData['tenantId']);
        $stmt->bindParam(':customerAdminFee', $inputData['tenantAdminFees']['customerAdminFee']);
        $stmt->bindParam(':tenantAdminFee', $inputData['tenantAdminFees']['tenantAdminFee']);
        $stmt->bindParam(':isActive', $inputData['tenantAdminFees']['isActive']);
        $stmt->execute();
        $lastInsertId = $pdo->lastInsertId();

    return [
        'status' => 'success',
        'message' => 'Tenant admin fees inserted successfully',
        'lastInsertId' => $lastInsertId
    ];
}
?>

