# Frontend API Interface Specification

---

## Overview
This document outlines the frontend API contract required to interact with the backend PHP application. All operations are triggered via a single endpoint (function.php) using POST or GET requests, with the action defined in the action field. Role-based access is enforced via session variables.

---

## Role Definitions

```sql
INSERT INTO `user_role` (`pk_role_id`, `role_name`, `scope`) VALUES
(1, 'SystemAdmin', 'Platform'),
(2, 'SupportAgent', 'Platform'),
(3, 'ContentModerator', 'Platform'),
(4, 'BusinessOwner', 'Tenant'),
(5, 'BusinessManager', 'Tenant'),
(6, 'Staff', 'Tenant'),
(7, 'Operator', 'Tenant'),
(8, 'Customer', 'Public');
```

## API Summary Table

| Action                      | Method | Required Role     | Payload Object         | Description                                 |
|----------------------------|--------|--------------------|-------------------------|---------------------------------------------|
| `newTenant`                | POST   | 1                  | `tenant`                | Create a new tenant                         |
| `updateTenant`             | POST   | 1, 3               | `tenant`                | Update an existing tenant                   |
| `deleteTenant`             | POST   | 1                  | `tenant`                | Soft-delete a tenant                        |
| `viewTenantSysAdmin`       | GET    | 1, 2, 3            | `tenantId`              | View tenant detail (sysadmin)               |
| `viewTenantOwner`          | GET    | 4                  | `tenantId`              | View tenant detail (owner view)             |
| `viewTenantPublic`         | GET    | 1, 2, 3            | `tenantId`              | View tenant detail (public view)            |
| `addNewTenantTheme`        | POST   | 1, 2, 3, 4, 5      | `tenantTheme`           | Add a new theme to a tenant                 |
| `updateTenantTheme`        | POST   | 1, 2, 3, 4, 5      | `tenantTheme`           | Update an existing theme                    |
| `viewThemeSysAdmin`        | GET    | 1, 2, 3            | `tenantId`              | View all themes (sysadmin view)             |
| `viewThemeOwner`           | GET    | 4, 5               | `tenantId` or `themeId` | View themes (owner view)                    |
| `viewThemePublic`          | GET    | Any                | `themeId`               | View theme detail (public view)             |
| `insertNewTenantContact`   | POST   | 4, 5               | `tenantContact`         | Add a new tenant contact (Owner)            |
| `insertNewTenantContactAdmin` | POST | 1, 2, 3            | `tenantContact`         | Add a new tenant contact (Admin)            |
| `viewTenantContactSysAdmin`| GET    | 1, 2, 3            | `tenantId`              | View contacts (SysAdmin)                    |
| `viewTenantContactOwner`   | GET    | 4, 5               | `tenantId`              | View contacts (Owner)                       |
| `viewTenantContactPublic`  | GET    | Any                | `tenantId`              | View public contact records                 |
| `updateTenantContact`      | POST   | 1, 2, 3, 4, 5      | `tenantContact`         | Update a tenant contact                     |
| `deleteTenantContact`      | POST   | 1, 2, 3, 4, 5      | `tenantContact`         | Soft-delete a contact                       |
| `insertNewBuisnessHours`   | POST   | 1, 2, 3, 4, 5      | `tenantBusinessHours`   | Insert or bulk insert business hours        |
| `viewBusinessHoursPublic`  | GET    | Any                | `tenantId`              | Retrieve public business hours              |
| `viewBusinessHoursOwner`   | GET    | 4, 5, 6            | `tenantId`              | View business hours (owner view)            |
| `viewBusinessHoursSysAdmin`| GET    | 1                  | `tenantId`              | View business hours (sysadmin view)         |
| `updateBusinessHours`      | POST   | 2, 4, 5            | `tenantBusinessHours`   | Update business hours                       |
| `deleteBusinessHours`      | POST   | 2, 4, 5            | `tenantBusinessHours`   | Soft-delete a business hour                 |
| `viewTenantChangeHistory`  | GET    | 1, 2, 3            | `tenantId`              | View full change history of a tenant        |
| `viewAllTenantsSuperAdmin`  | GET    | 1, 2, 3            |                   | View full view all tenants info high level        |

---

## Payload Structures

### `newTenant`
```json
{
  "action": "newTenant",
  "tenant": {
    "companyName": "string",
    "domain": "string",
    "companyNumber": "string",
    "foodHygieneId": "string",
    "primaryContactName": "string",
    "tenantStatus": "active|pending",
    "parentId": optional
  }
}
```

### `updateTenant`
```json
{
  "action": "updateTenant",
  "tenant": {
    "tenantId": int,
    "companyName": optional,
    "domain": optional,
    "companyNumber": optional,
    "foodHygieneId": optional,
    "primaryContactName": optional,
    "tenantStatus": optional,
    "isDeleted": optional,
    "parentId": optional
  }
}
```

### `deleteTenant`
```json
{
  "action": "deleteTenant",
  "tenant": {
    "tenantId": int
  }
}
```

### `viewTenantSysAdmin`
```json
{
  "action": "viewTenantSysAdmin",
  "tenantId": int
}
```

### `viewTenantOwner`
```json
{
  "action": "viewTenantOwner",
  "tenantId": int
}
```

### `viewTenantPublic`
```json
{
  "action": "viewTenantPublic",
  "tenantId": int
}
```

### `addNewTenantTheme`
```json
{
  "action": "addNewTenantTheme",
  "tenantTheme": {
    "tenantId": int,
    "themeName": "string",
    "primaryColor": "#000000",
    "secondaryColor": "#000000",
    "textColor": "#000000",
    "secondaryTextColor": "#000000",
    "accentColor": "#000000",
    "backgroundColor": "#000000",
    "buttonColor": "#000000"
  }
}
```

### `updateTenantTheme`
```json
{
  "action": "updateTenantTheme",
  "tenantTheme": {
    "themeId": int,
    "isApplied": "yes|no",
    "themeStatus": "active|draft|deleted",
    "isDeleted": optional
  }
}
```

### `viewThemeSysAdmin`
```json
{
  "action": "viewThemeSysAdmin",
  "tenantId": int
}
```

### `viewThemeOwner`
```json
{
  "action": "viewThemeOwner",
  "tenantId": int // or "themeId": int
}
```

### `viewThemePublic`
```json
{
  "action": "viewThemePublic",
  "themeId": int
}
```

### `insertNewTenantContact`
```json
{
  "action": "insertNewTenantContact",
  "tenantContact": {
    "tenantId": int,
    "label": "string",
    "contactType": "Email|Name|Phone|...",
    "contactValue": "string",
    "isPrimary": "yes|no"
  }
}
```

### `insertNewTenantContactAdmin`
```json
{
  "action": "insertNewTenantContactAdmin",
  "tenantContact": {
    "tenantId": int,
    "userId": int,
    "label": "string",
    "contactType": "string",
    "contactValue": "string",
    "isPrimary": "yes|no"
  }
}
```

### `viewTenantContactSysAdmin`
```json
{
  "action": "viewTenantContactSysAdmin",
  "tenantId": int
}
```

### `viewTenantContactOwner`
```json
{
  "action": "viewTenantContactOwner",
  "tenantId": int
}
```

### `viewTenantContactPublic`
```json
{
  "action": "viewTenantContactPublic",
  "tenantId": int
}
```

### `updateTenantContact`
```json
{
  "action": "updateTenantContact",
  "tenantContact": {
    "contactId": int,
    "tenantId": optional,
    "userId": optional,
    "label": "string",
    "contactType": "string",
    "contactValue": "string",
    "isPrimary": "yes|no"
  }
}
```

### `deleteTenantContact`
```json
{
  "action": "deleteTenantContact",
  "tenantContact": {
    "contactId": int,
    "tenantId": optional,
    "userId": optional,
    "isDeleted": 1
  }
}
```

### `insertNewBuisnessHours`
```json
{
  "action": "insertNewBuisnessHours",
  "tenantBusinessHours": [
    {
      "tenantId": int,
      "hoursCategory": "store|delivery|special",
      "openingDay": "Monday|...",
      "openTime": "HH:MM:SS",
      "closeTime": "HH:MM:SS",
      "isClosed": optional,
      "specificDate": optional (for special)
    }
  ]
}
```

### `viewBusinessHoursPublic`
```json
{
  "action": "viewBusinessHoursPublic",
  "tenantId": int
}
```

### `viewBusinessHoursOwner`
```json
{
  "action": "viewBusinessHoursOwner",
  "tenantId": int
}
```

### `viewBusinessHoursSysAdmin`
```json
{
  "action": "viewBusinessHoursSysAdmin",
  "tenantId": int
}
```

### `updateBusinessHours`
```json
{
  "action": "updateBusinessHours",
  "tenantBusinessHours": {
    "businessHoursId": int,
    "hoursCategory": "store|delivery|special",
    "openingDay": optional,
    "specificDate": optional,
    "openTime": optional,
    "closeTime": optional,
    "isClosed": optional,
    "isDeleted": optional
  }
}
```

### `deleteBusinessHours`
```json
{
  "action": "deleteBusinessHours",
  "tenantBusinessHours": {
    "businessHoursId": int,
    "isDeleted": 1
  }
}
```

### `viewTenantChangeHistory`
```json
{
  "action": "viewTenantChangeHistory",
  "tenantId": int
}
```
### `viewAllTenantsSuperAdmin`
```json
{
  "action": "viewAllTenantsSuperAdmin"
}
```
---

## Response Format

### Success
```json
{
  "status": "success",
  "message": "string",
  // additional data if applicable
}
```

### Failure
```json
{
  "status": "failed",
  "message": "Error message string"
}
```

---

## Frontend Implementation Notes

- Ensure session authentication is maintained
- Validate required fields before submission
- Show proper error/success messaging
- Use roleValue to determine UI behavior (edit/delete/submit availability)
- Display conditional components based on `isDeleted`, `isPrimary`, and `themeStatus`

---

_End of Document_

