<?php
function sendRequestEmail(array $inputData): array{
    $companyInfo = getCompanyInfo();
    $companyName = $companyInfo['companyName'];

    // Handle preflight request
    if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
        header("Access-Control-Allow-Origin: *");
        header("Access-Control-Allow-Methods: POST");
        header("Access-Control-Allow-Headers: content-type");
        exit;
    }

    // Only allow POST requests
    if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
        header("Allow: POST", true, 405);
        exit;
    }

    header("Access-Control-Allow-Origin: *");
    header("Content-Type: application/json; charset=UTF-8");

    // Initialize response data
    $data = ['result' => []];

    // ---------------------------
    // OPTIONAL cheap spam hardening: Honeypot BEFORE Turnstile
    // ---------------------------
    // Add a hidden field on frontend, e.g. "companyWebsite", should be blank.
    /*
    if (!empty($inputData['companyWebsite'])) {
        error_log('Honeypot triggered: ' . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
        http_response_code(400);
        $data['result']['status'] = 'error';
        $data['result']['message'] = 'Invalid submission.';
        $data['result']['http_response_code'] = 400;
        return $data;
    }
    */

    // ---------------------------
    // TURNSTILE SPAM GATE (REJECT HERE)
    // verify_turnstile returns: [$inputData, $okBool, $httpCode]
    // ---------------------------
    [$inputData, $ok, $verifyHttpCode] = verify_turnstile($inputData);

    if (!$ok) {
        // Safe logging: log error codes (no token, no secret)
        $codes = implode(',', $inputData['turnstileVerification']['errorCodes'] ?? []);
        error_log('Turnstile failed: ip=' . ($_SERVER['REMOTE_ADDR'] ?? 'unknown') . ' codes=' . $codes);

        http_response_code($verifyHttpCode);
        $data['result']['status'] = 'error';
        $data['result']['http_response_code'] = $verifyHttpCode;

        if ($verifyHttpCode === 400) {
            $data['result']['message'] = 'Missing Turnstile token.';
        } elseif ($verifyHttpCode === 500) {
            $data['result']['message'] = 'Server misconfiguration.';
        } elseif ($verifyHttpCode === 502) {
            $data['result']['message'] = 'Verification temporarily unavailable.';
        } else { // 403
            $data['result']['message'] = 'Turnstile verification failed.';
        }

        return $data;
    }

    // ---------------------------
    // Continue with your normal processing (only after Turnstile passes)
    // ---------------------------

    // Validate and sanitize input data
    $name     = htmlspecialchars(strip_tags((string)($inputData['name'] ?? '')));
    $emailRaw = (string)($inputData['email'] ?? '');
    $email    = filter_var($emailRaw, FILTER_VALIDATE_EMAIL);
    $phone    = htmlspecialchars(strip_tags((string)($inputData['phone'] ?? '')));
    $enquiry  = htmlspecialchars(strip_tags((string)($inputData['message'] ?? '')));
    $location = htmlspecialchars(strip_tags((string)($inputData['location'] ?? '')));
    $timescale= htmlspecialchars(strip_tags((string)($inputData['timescale'] ?? '')));

    // Basic required checks (optional but recommended)
    if ($name === '' || !$email || $enquiry === '') {
        http_response_code(422);
        $data['result']['status'] = 'error';
        $data['result']['message'] = 'Missing or invalid required fields.';
        $data['result']['http_response_code'] = 422;
        return $data;
    }

    // Services
    $services2 = (string)($inputData['services'] ?? '');
    $services = '';
    if ($services2 !== '') {
        $services3 = explode(',', $services2);
        foreach ($services3 as $service4) {
            $ser  = trim(html_entity_decode($service4), '"');
            $ser2 = str_replace(["[", "]"], "", $ser);
            $services .= $ser2 . ", ";
        }
        $services = rtrim($services, ", ");
        $services = str_replace('"', "", $services);
    }

    // Image info
    $img_info = '';
    if (!empty($inputData['imgList'])) {
        $image_info = json_decode(stripslashes(html_entity_decode((string)$inputData['imgList'])));
        $tmp = "</br><p><strong>Please see attached images</strong></p></br>";
        $i = 0;

        if (is_array($image_info)) {
            foreach ($image_info as $file) {
                $i++;
                $url = isset($file->url) ? (string)$file->url : '';
                if ($url === '') continue;

                $tmp .= "<p><a href='" . htmlspecialchars($url, ENT_QUOTES, 'UTF-8') . "' target='_blank'>Image {$i}</a></p>";
                $tmp .= "</br>";
            }
            $img_info = $tmp;
        }
    }

    // Get server domain for the no-reply email address
    $domain = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $noreplyEmail = "no-reply@$domain";

    // Build admin email content
    $servicesList = $services ? "<p><strong>Services:</strong> " . $services . "</p>" : "";
    $conextra     = $enquiry ? "<p><strong>Enquiry:</strong><br>" . nl2br($enquiry) . "</p>" : "";
    $phoneLink    = $phone ? "<p><strong>Contact By Phone:</strong> <a href=\"tel:$phone\">$phone</a></p>" : "";
    $location1    = $location ? "<p><strong>Location:</strong> $location</p>" : "";
    $timescale1   = $timescale ? "<p><strong>Timescale:</strong> $timescale</p>" : "";

    $adminRecipient = $companyInfo['adminRecipient'];
    $adminSubject   = "New Enquiry";

    $adminHeaders   = "From: $noreplyEmail\r\n";
    $adminHeaders  .= "Reply-To: $email\r\n";
    $adminHeaders  .= "Content-type: text/html; charset=UTF-8\r\n";
    $adminHeaders  .= "Bcc:admin@prodigydesign.co.uk\r\n";

    $adminMessage = "
    <html>
    <head>
        <title>$adminSubject</title>
    </head>
    <body>
        <p>Hi Admin,</p>
        <p>We have a customer, <strong>$name</strong></p>
        $conextra
        $img_info
        $servicesList
        <p><strong>Contact By Email:</strong> $email</p>
        $phoneLink
        $location1
        $timescale1
        <p>Email generated from the website.</p>
        <p><em>Replies to this mailbox are not monitored.</em></p>
        <p>" . date('Y-m-d H:i:s') . "</p>
    </body>
    </html>
    ";

    // Send the admin email
    $adminSuccess = mail($adminRecipient, $adminSubject, $adminMessage, $adminHeaders);

    if (!$adminSuccess) {
        http_response_code(500);
        $data['result']['status'] = 'error';
        $data['result']['message'] = 'Failed to send admin email.';
        $data['result']['http_response_code'] = 500;
        return $data;
    }

    // Send thank you email to user (you already have this)
    $userSuccess = sendThankYouEmail($inputData, $companyName);

    // Success response (even if thank you email fails, admin email is the core)
    http_response_code(200);
    $data['result']['status'] = 'success';
    $data['result']['message'] = 'Admin email sent successfully.';
    $data['result']['http_response_code'] = 200;
    // Optional: include this if you want visibility
    // $data['result']['thank_you_sent'] = (bool)$userSuccess;

    return $data;
}

function sendThankYouEmail($inputData, $companyName) {
    // Get server domain for the no-reply email address
    $domain = $_SERVER['HTTP_HOST'];
    $noreplyEmail = "no-reply@$domain";

    // Extract user details from input data
    $name = htmlspecialchars(strip_tags($inputData['name']));
    $email = filter_var($inputData['email'], FILTER_VALIDATE_EMAIL);

    // Build user email content
    $userSubject = 'Thank you for your enquiry';
    $userHeaders = "From: $noreplyEmail\r\n";
    $userHeaders .= "Content-type: text/html; charset=UTF-8\r\n";

    $userMessage = "
    <html>
    <head>
        <title>$userSubject</title>
    </head>
    <body>
        <p>Dear $name,</p>
        <p>Thank you for your enquiry to $companyName. Someone from our team will be in contact with you soon.</p>
        <p><em>Replies to this mailbox are not monitored.</em></p>
    </body>
    </html>
    ";

    // Send the user email
    return mail($email, $userSubject, $userMessage, $userHeaders);
}

function unsetTurnstileVerification(array &$inputData): void {
    if (isset($inputData['secure']['turnstileSecret'])) {
        unset($inputData['secure']['turnstileSecret']);
    }
}

function verify_turnstile(array $inputData): array{
    $token = isset($inputData['turnstileToken']) ? (string)$inputData['turnstileToken'] : '';

    // Prepare verification result container (always set it)
    $inputData['turnstileVerification'] = [
        'ok' => false,
        'errorCodes' => [],
        'action' => null,
        'hostname' => null,
        'challengeTs' => null,
        'httpCode' => null,
    ];


    if ($token === '') {
        $inputData['turnstileVerification']['errorCodes'] = ['missing_token'];
        return [$inputData, false, 400];
    }

    turnstileSecret($inputData);

    $secret = (string)($inputData['secure']['turnstileSecret'] ?? '');
    unsetTurnstileVerification($inputData);
    if ($secret === '') {
        $inputData['turnstileVerification']['errorCodes'] = ['missing_secret'];
        return [$inputData, false, 500];
    }

    $remoteIp = $_SERVER['REMOTE_ADDR'] ?? null;

    // Optional action binding: only enforce if you set it
    // For your payload you have "action": "emailRequest"
    // Only enforce if you also configured Turnstile widget/action on the frontend.
    $expectedAction = null; // or 'emailRequest'

    $url = 'https://challenges.cloudflare.com/turnstile/v0/siteverify';
    $timeoutSeconds = 5;

    $fields = [
        'secret'   => $secret,
        'response' => $token,
    ];
    if ($remoteIp) {
        $fields['remoteip'] = $remoteIp;
    }

    $ch = curl_init($url);
    if ($ch === false) {
        $inputData['turnstileVerification']['errorCodes'] = ['curl_init_failed'];
        return [$inputData, false, 502];
    }

    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => http_build_query($fields),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => $timeoutSeconds,
        CURLOPT_CONNECTTIMEOUT => $timeoutSeconds,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    $body = curl_exec($ch);
    $curlErrNo = curl_errno($ch);
    $curlErr   = curl_error($ch);
    $httpCode  = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $inputData['turnstileVerification']['httpCode'] = $httpCode;

    if ($body === false || $curlErrNo !== 0) {
        // Don’t expose curl error details to client; keep for logs only if needed
        $inputData['turnstileVerification']['errorCodes'] = ['curl_error'];
        error_log('Turnstile curl error: ' . $curlErrNo . ' ' . $curlErr);
        return [$inputData, false, 502];
    }

    $json = json_decode($body, true);
    if (!is_array($json)) {
        $inputData['turnstileVerification']['errorCodes'] = ['invalid_json_response'];
        return [$inputData, false, 502];
    }

    $success = (bool)($json['success'] ?? false);
    $errorCodes = array_values((array)($json['error-codes'] ?? []));

    $action = isset($json['action']) ? (string)$json['action'] : null;

    // Cloudflare non-2xx => failure
    if ($httpCode < 200 || $httpCode >= 300) {
        $success = false;
    }

    // Optional action mismatch enforcement
    if ($success && $expectedAction !== null && $action !== $expectedAction) {
        $success = false;
        $errorCodes[] = 'action_mismatch';
    }

    // Write safe results back to inputData
    $inputData['turnstileVerification']['ok'] = $success;
    $inputData['turnstileVerification']['errorCodes'] = $errorCodes;
    $inputData['turnstileVerification']['action'] = $action;
    $inputData['turnstileVerification']['hostname'] = isset($json['hostname']) ? (string)$json['hostname'] : null;
    $inputData['turnstileVerification']['challengeTs'] = isset($json['challenge_ts']) ? (string)$json['challenge_ts'] : null;

    // Choose status: 200 if ok, 403 if challenge failed
    if ($success) {
        return [$inputData, true, 200];
    }
    return [$inputData, false, 403];
}

?>