-- Menu Table
CREATE TABLE menu_menus (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    name TEXT NOT NULL,
    description TEXT,
    is_active INTEGER DEFAULT 1,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Categories
CREATE TABLE menu_categories (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    menu_id INTEGER NOT NULL,
    parent_id INTEGER,
    name TEXT NOT NULL,
    description TEXT,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (menu_id) REFERENCES menu_menus(id),
    FOREIGN KEY (parent_id) REFERENCES menu_categories(id)
);

-- Items
CREATE TABLE menu_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    item_name TEXT NOT NULL,
    item_description TEXT,
    base_price REAL,
    nutrition_id INTEGER,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)
);

-- Category-Item Linking
CREATE TABLE menu_category_items (
    category_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (category_id, item_id),
    FOREIGN KEY (category_id) REFERENCES menu_categories(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- Nutrition Profiles
CREATE TABLE menu_nutrition_profiles (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    calories INTEGER DEFAULT 0,
    protein_grams REAL DEFAULT 0,
    fat_grams REAL DEFAULT 0,
    carbs_grams REAL DEFAULT 0,
    sugar_grams REAL DEFAULT 0,
    salt_grams REAL DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Option Groups
CREATE TABLE menu_option_groups (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    item_id INTEGER NOT NULL,
    name TEXT NOT NULL,
    description TEXT,
    min_select INTEGER DEFAULT 0,
    max_select INTEGER DEFAULT 1,
    is_required INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- Option Items
CREATE TABLE menu_option_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    group_id INTEGER NOT NULL,
    name TEXT NOT NULL,
    description TEXT,
    price_adjustment REAL DEFAULT 0,
    nutrition_id INTEGER,
    is_default INTEGER DEFAULT 0,
    allow_toggle_only INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (group_id) REFERENCES menu_option_groups(id),
    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)
);

-- Meals
CREATE TABLE menu_meals (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    category_id INTEGER NOT NULL,
    name TEXT NOT NULL,
    description TEXT,
    base_price REAL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id)
);

-- Meal Parts
CREATE TABLE menu_meal_parts (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    meal_id INTEGER NOT NULL,
    name TEXT NOT NULL,
    description TEXT,
    min_select INTEGER DEFAULT 1,
    max_select INTEGER DEFAULT 1,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (meal_id) REFERENCES menu_meals(id)
);

-- Meal Part Items
CREATE TABLE menu_meal_part_items (
    part_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (part_id, item_id),
    FOREIGN KEY (part_id) REFERENCES menu_meal_parts(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);


-- ----------------------------
-- Table structure for menu_history
-- ----------------------------
CREATE TABLE menu_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id INT NOT NULL,
    action_type ENUM('updated', 'deleted') NOT NULL,
    table_of_source VARCHAR(50) NOT NULL,
    old_value TEXT DEFAULT NULL,
    action_by INT NOT NULL,
    action_at DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- ----------------------------
-- View: menu_active_menus_per_tenant
-- Shows the currently active menu for each tenant
-- ----------------------------
CREATE OR REPLACE VIEW menu_active_menus_per_tenant AS
SELECT *
FROM menu_menus
WHERE is_active = 1 AND is_deleted = 0;


-- ----------------------------
-- Trigger: Ensure only one active menu per tenant
-- ----------------------------
DELIMITER $$

CREATE TRIGGER trg_only_one_active_menu
BEFORE INSERT ON menu_menus
FOR EACH ROW
WHEN NEW.is_active = 1
BEGIN
    UPDATE menu_menus
    SET is_active = 0
    WHERE tenant_id = NEW.tenant_id AND is_active = 1 AND is_deleted = 0;
END $$

CREATE TRIGGER trg_only_one_active_menu_update
BEFORE UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.is_active = 1 AND OLD.is_active != 1
BEGIN
    UPDATE menu_menus
    SET is_active = 0
    WHERE tenant_id = NEW.tenant_id AND id != NEW.id AND is_active = 1 AND is_deleted = 0;
END $$


DELIMITER ;

-- ----------------------------
-- Trigger: Log updates and deletes for menu_items
-- ----------------------------
DELIMITER $$

CREATE TRIGGER trg_menu_items_update
AFTER UPDATE ON menu_items
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id,
        action_type,
        table_of_source,
        old_value,
        action_by
    )
    VALUES (
        OLD.menu_id,
        'updated',
        'menu_items',
        JSON_OBJECT(
            'id', OLD.id,
            'name', OLD.name,
            'base_price', OLD.base_price
        ),
        0
    );
END$$

CREATE TRIGGER trg_menu_items_delete
BEFORE DELETE ON menu_items
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id,
        action_type,
        table_of_source,
        old_value,
        action_by
    )
    VALUES (
        OLD.menu_id,
        'deleted',
        'menu_items',
        JSON_OBJECT(
            'id', OLD.id,
            'name', OLD.name,
            'base_price', OLD.base_price
        ),
        0
    );
END$$

DELIMITER $$

CREATE TRIGGER trg_menu_menus_is_deleted
AFTER UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.is_deleted != OLD.is_deleted
BEGIN
    UPDATE menu_categories
    SET is_deleted = NEW.is_deleted
    WHERE menu_id = NEW.id;
END$$


DELIMITER ;


DELIMITER ;
