-- Guest User Table
CREATE TABLE food_order_guest_user (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    f_name TEXT,
    l_name TEXT,
    phone_number TEXT,
    user_email TEXT,
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now'))
);

-- Address Table
CREATE TABLE food_order_address ( -- once a value is added to this table, it cannot be deleted, or updated as this links to an order, we can only add new addresses and deleted after an order is older than a set time based on the country's laws of data retention
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    address_line_1 TEXT,
    address_line_2 TEXT,
    city TEXT,
    county TEXT,
    country TEXT,
    postal_code TEXT,
    check_sum TEXT UNIQUE,
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now'))
);

-- Order Table
CREATE TABLE "food_order" (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    public_order_code TEXT,
    user_id INTEGER,
    guest_user_id INTEGER,
    tenant_id INTEGER NOT NULL,
    delivery_address_id INTEGER,
    billing_address_id INTEGER,
    order_status TEXT CHECK(order_status IN ('pending', 'confirmed', 'cooking', 'delivery', 'completed', 'cancelled')) DEFAULT 'pending',
    order_type TEXT CHECK(order_type IN ('delivery', 'pickup', 'dine-in', 'collection')) NOT NULL,
    total_order_price REAL NOT NULL DEFAULT 0.00,
    order_full_refund REAL DEFAULT 0.00,
    total_order_modified_price REAL,
    original_total_order_modified_price REAL,
    total_order_modified_reason TEXT,
    payment_status TEXT CHECK(payment_status IN ('unpaid', 'payment_intent', 'paid', 'refunded', 'partial_refund', 'payment_pre_auth')) DEFAULT 'unpaid',
    delivery_fee REAL DEFAULT 0.00,
    order_fee REAL DEFAULT 0.00,
    platform_fee REAL DEFAULT 0.00,
    stripe_fee_total REAL DEFAULT 0.00,
    small_order_fee REAL DEFAULT 0.00,
    comment TEXT,
    tenant_comment TEXT,
    currency TEXT DEFAULT 'GBP',
    transaction_id TEXT,
    refunded_id TEXT,
    payment_intent_id TEXT,
    stripe_receipt_url TEXT,
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now')),
    -- Note: SQLite does not support CHECK on multiple columns, so enforce guest/user exclusivity in app logic
    FOREIGN KEY (guest_user_id) REFERENCES food_order_guest_user(id),
    FOREIGN KEY (delivery_address_id) REFERENCES food_order_address(id),
    FOREIGN KEY (billing_address_id) REFERENCES food_order_address(id),
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id)
);

-- Item Table
CREATE TABLE food_order_item (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    order_id INTEGER NOT NULL,
    menu_item_id INTEGER NOT NULL,
    item_name TEXT,
    item_base_price REAL,
    item_modified_price REAL,
    item_modified_reason TEXT,
    item_quantity INTEGER NOT NULL DEFAULT 1,
    item_note TEXT,
    item_status TEXT CHECK(item_status IN ('pending', 'prepared', 'served', 'cancelled', 'refunded', 'partial_refund')) DEFAULT 'pending',
    item_tenant_comment TEXT,
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now')),
    FOREIGN KEY (order_id) REFERENCES food_order(id) ON DELETE CASCADE
);

-- Extra Table
CREATE TABLE food_order_extra (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    item_id INTEGER NOT NULL,
    menu_extra_id INTEGER,
    extra_name TEXT,
    extra_price REAL,
    extra_quantity INTEGER NOT NULL DEFAULT 1,
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now')),
    FOREIGN KEY (item_id) REFERENCES food_order_item(id) ON DELETE CASCADE
);

-- User Address Table
CREATE TABLE food_order_user_address (
    id          INTEGER PRIMARY KEY AUTOINCREMENT,
    user_id     INTEGER NOT NULL,
    address_id  INTEGER NOT NULL,
    is_default  INT DEFAULT 0,
    is_active   INT DEFAULT 1,
    is_deleted  INT DEFAULT 0,
    created_at  TEXT DEFAULT (DATETIME('now')),
    updated_at  TEXT DEFAULT (DATETIME('now')),
    FOREIGN KEY (user_id)       REFERENCES user_login(pk_user_id),
    FOREIGN KEY (address_id)    REFERENCES food_order_address(id)
);

-- food_order_history
CREATE TABLE food_order_history (
    id INTEGER  PRIMARY KEY AUTOINCREMENT,
    order_id INTEGER NOT NULL,
    tenant_id INTEGER NOT NULL,
    change_type CHECK(change_type IN('create', 'update', 'delete', 'refund')) NOT NULL,
    table_of_source TEXT NOT NULL, -- Name of the table where the change occurred
    change_data JSON NOT NULL,
    changed_by INTEGER NOT NULL, -- User ID of the person who made the change
    created_at TEXT DEFAULT (DATETIME('now')),
    FOREIGN KEY (order_id)      REFERENCES food_order(id) ON DELETE CASCADE
);

CREATE TABLE food_address_to_tenant (
    id INTEGER  PRIMARY KEY AUTOINCREMENT,
    tenant_id   INTEGER NOT NULL,
    address_id  INTEGER NOT NULL,
    distance    REAL,
    postal_code TEXT,
    created_at  TEXT DEFAULT (DATETIME('now')),
    updated_at  TEXT DEFAULT (DATETIME('now')),
    FOREIGN KEY (address_id) REFERENCES food_order_address(id),
    FOREIGN KEY (tenant_id) REFERENCES  tenant(id)
);

-- ========================================
-- TERMS OF SERVICE SCHEMA
-- ========================================

-- Terms of Service Content Table
-- Stores the actual terms content, versioned and tenant-specific
CREATE TABLE food_order_terms_of_service (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    versions TEXT NOT NULL,                     -- e.g., "1.0", "2.1", "2024-01-15"
    title TEXT DEFAULT 'Terms of Service',
    content TEXT NOT NULL,                      -- HTML or markdown content
    effective_date TEXT NOT NULL,               -- When these terms become active (DATE format)
    expiry_datee TEXT DEFAULT NULL,             -- Optional: when terms expire (DATE format)
    is_active INTEGER DEFAULT 0,                -- Only one active version per tenant (0/1 for boolean)
    is_deleted INTEGER DEFAULT 0,               -- Soft delete flag (0/1 for boolean)
    requires_acceptance INTEGER DEFAULT 1,      -- Some updates might not require re-acceptance (0/1 for boolean)
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now'))
);

-- Terms Acceptance Tracking Table
-- Tracks when users (registered or guest) accept terms
CREATE TABLE food_order_terms_acceptance (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    terms_id INTEGER NOT NULL,                 -- Links to specific terms version
    tenant_id INTEGER NOT NULL,
    
    -- User identification (matches food_order table pattern)
    user_id INTEGER DEFAULT NULL,              -- Registered user
    guest_user_id INTEGER DEFAULT NULL,        -- Guest user
    order_id INTEGER DEFAULT NULL,             -- Optional: link to specific order
    
    -- Acceptance details
    accepted_at TEXT DEFAULT (DATETIME('now')),
    ip_address TEXT,                           -- IPv4 or IPv6
    user_agent TEXT,                           -- Browser/device info
    acceptance_method TEXT CHECK(acceptance_method IN ('checkbox', 'button', 'implicit', 'api')) DEFAULT 'checkbox',
    
    -- Legal/audit trail
    terms_version TEXT,                        -- Denormalized for easy reporting
    terms_checksum TEXT,                       -- Hash of terms content when accepted
    
    created_at TEXT DEFAULT (DATETIME('now')),
    
    FOREIGN KEY (terms_id) REFERENCES food_order_terms_of_service(id),
    FOREIGN KEY (order_id) REFERENCES food_order(id) ON DELETE SET NULL,
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE CASCADE,
    FOREIGN KEY (guest_user_id) REFERENCES food_order_guest_user(id) ON DELETE CASCADE
    
    -- Note: SQLite constraint for user XOR guest enforced in application logic
);

-- Terms Acceptance Summary Table
-- Quick lookup for checking if user has accepted current terms
CREATE TABLE food_order_terms_acceptance_summary (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    
    -- User identification
    user_id INTEGER DEFAULT NULL,
    guest_user_id INTEGER DEFAULT NULL,
    
    -- Current acceptance status
    latest_terms_version TEXT,
    latest_accepted_at TEXT,
    requires_new_acceptance INTEGER DEFAULT 0, -- 0/1 for boolean
    
    created_at TEXT DEFAULT (DATETIME('now')),
    updated_at TEXT DEFAULT (DATETIME('now')),
    
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE CASCADE,
    FOREIGN KEY (guest_user_id) REFERENCES food_order_guest_user(id) ON DELETE CASCADE
    
    -- Note: SQLite constraints for uniqueness and user XOR guest enforced in application logic
);
