<?php
    function updateMenuTable(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $inputData['step'][] = 'validate input for update';
        // Step 1: Validate table, id, and data structure
        $table = $inputData['table'] ?? null;
        $roleId = $inputData['roleId'] ?? 0;
        $dataKey = $inputData['dataKey'];
        $data = $inputData[$dataKey] ?? [];
        $idField = $inputData['idField'] ?? 'id'; // Default to 'id' unless specified
        $tableName = $inputData['table'] ?? null;
        $table = toSnakeCase($table);
        if (!$table) {
            return ['status' => 'failed', 'message' => 'Table not specified'];
        }

        $inputData['step'][] = 'get column mapping and permissions';
        $map = menuTableColumnMapping($table, $roleId, $inputData['recordType'] ?? null);
        $columnMap = $map['filtered'] ?? null;

        $inputData['step'][] = 'check if column mapping is valid';
        if (!$columnMap) {
            //if debug mode is on, return the map
            if (isset($inputData['debug']) && $inputData['debug'] === 'CSD') {
                return ['status' => 'failed', 'message' => 'Invalid table or insufficient permissions for table: ' . $table. ' for role: ' . $roleId];
            }else{
                return ['status' => 'failed', 'message' => 'Invalid table or insufficient permissions'];
            }
        }
        $inputData['step'][] = 'build update statement parts';
        $updateParts = [];
        $bindParams = [];
        $errors = [];

        // Step 2: Build SQL SET clause and bind parameters
        $inputData['step'][] = 'build update statement parts';
        foreach ($columnMap as $key => $meta) {
            $inputData['step'][] = "processing field: $key";
            // Skip fields with optionalUpdate === null (cannot update)
            if (array_key_exists('optionalUpdate', $meta) && $meta['optionalUpdate'] === null) {
                continue;
            }

            $dbField = $meta['mapping'] ?? null;
            if (!$dbField) {
                continue;
            }

            // ⛔️ CHANGE: never put the ID field into the SET clause
            if ($key === $idField || $dbField === $idField) {
                continue;
            }

            if (isset($data[$key])) {
                // Check if the user role is allowed to update this field
                if (!in_array($roleId, $meta['updateRole'] ?? [])) {
                    $errors[] = "Role $roleId not allowed to update $key";
                    continue;
                }

                $updateParts[] = "$dbField = :$key";
                $bindParams[$key] = $data[$key];
            } else {
                if (!empty($meta['optionalUpdate']) && $meta['optionalUpdate'] == 1) {
                    // Field is optional for update, OK to skip
                    continue;
                }
                // If field is required and not present, error (optional, depends how strict you want update to be)
            }
        }

        $inputData['step'][] = 'final validation before executing update';
        if (!empty($errors)) {
            return [
                'status' => 'failed',
                'message' => implode(', ', $errors)
            ];
        }

        // If no fields to update, return error
        $inputData['step'][] = 'check if there are fields to update';
        if (empty($updateParts)) {
            if (isset($inputData['debug']) && $inputData['debug'] === 'CSD') {
                return ['status' => 'failed', 'message' => 'No valid fields to update ' . 'columnMap: ' . print_r($columnMap, true)];
            }else{
                return ['status' => 'failed', 'message' => 'No valid fields to update'];
            }
        }

        // Note: original had a check for "only ID field is being updated".
        // Since we now skip the ID from SET, that situation cannot occur anymore.

        $rowId = $data[$idField] ?? null;
        $inputData['step'][] = 'check if ID field value is provided';
        if ($rowId === null) {
            return [
                'status' => 'failed',
                'message' => "Missing $idField value for WHERE clause"
            ];
        }

        $inputData['step'][] = 'handle special cases before update';
        if($table == 'menu_menus'){
            if(isset($data['isActive']) && $data['isActive'] == 1){
                if(in_array('isActive', array_keys($columnMap))){    
                    $tempRes = disableAnyActive($inputData, $table); // empty string as may break update function
                    if($tempRes['status'] == 'failed'){  
                        return ['res' => $tempRes];
                    }
                }
                    
            }
        }
        
        $inputData['step'][] = 'prepare and execute update statement';
        // Build SQL
        $setClause = implode(', ', $updateParts);
        $whereClause = "{$idField} = :id";
        $sql = "UPDATE {$table} SET {$setClause} WHERE {$whereClause}";

        $stmt = $pdo->prepare($sql);

        $inputData['step'][] = 'bind parameters and execute statement';
        foreach ($bindParams as $param => $value) {
            $stmt->bindValue(":$param", $value);
        }
        // Bind ID only for WHERE
        $stmt->bindValue(":id", $rowId);

        $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging
        $stmt->execute();

        $inputData['step'][] = 'finalize update and return result';
        if ($stmt->rowCount() === 0) {
            $inputData['step'][] = 'no rows updated';
            return [
                'status' => 'failed',
                'message' => "No rows updated. Data may be identical or {$idField} not found."
            ];
        }

        $inputData['step'][] = 'update successful';
        return [
            'status' => 'success',
            'message' => 'Record updated successfully'
        ];
    }
    function deleteContent($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Step 1: Validate table, id, and data structure
        $res = validateUpdateInputStep($inputData);
        if ($res['status'] !== 'success') {
            $pdo->rollback();
            return $res;
        }

        $tableName = toSnakeCase($inputData['table']);
        
        //$tableName =   //$inputData['table'];
        $roleId = $inputData['roleId'];
        $dataKey = $inputData['dataKey'];
        $tableData = $inputData[$dataKey];
        $idField = $inputData['idField'];
        $idValue = $inputData['id'];
        $extraSetsSql = "  ";

        // Step 2: Check if record exists and is not already deleted
        $checkSql = "SELECT is_deleted FROM {$tableName} WHERE {$idField} = :id LIMIT 1";
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->bindValue(':id', $idValue);
        $checkStmt->execute();
        $record = $checkStmt->fetch(PDO::FETCH_ASSOC);

        if (!$record) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Record not found for ID {$idValue}"];
        }

        if (!empty($record['is_deleted'])) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Record with ID {$idValue} is already deleted"];
        }

        // Step 3: Check user permission to update (delete is an update of is_deleted field)
        $map = menuTableColumnMapping($tableName, $roleId);
        $columnMap = $map['filtered'] ?? null;
        
        if (!$columnMap || !isset($columnMap['isDeleted'])) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Delete permission configuration missing for {$tableName}"];
        }

        if (!in_array($roleId, $columnMap['isDeleted']['updateRole'] ?? [])) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "User role not permitted to delete this record"];
        }

        // Step 4: check if there is an is_active field and if it is set to 0 then do delete
        if(in_array('isActive', array_keys($columnMap))){
            $extraSetsSql .= " , is_active = 0 ";
        }

        // Step 5: Perform the soft delete
        $sql = "UPDATE {$tableName} SET is_deleted = 1 {$extraSetsSql} WHERE {$idField} = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(":id", $idValue);
        
        $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging\
        $stmt->execute();

        if ($stmt->rowCount() === 0) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Failed to delete record with ID {$idValue}"];
        }

        return ['status' => 'success', 'message' => 'Record marked as deleted successfully'];
    }
    function disableAnyActive($inputData, $tableName){
        $tenantId = $inputData['tenantId'];
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Step 1: check if there are any reccords that match our condition

        $sql = "select count(*) as count from $tableName where tenant_id = :tenantId and is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId);
        $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging\
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $rowCount = $result['count'];
        if($rowCount == 0){
            return [
                'status' => 'success',
                'message' => 'No rows found.',
                'tenantId' => $tenantId,
                'tableName' => $tableName
            ];
        }else{
            // Step 2: update all recoreds to is_applied = 'no' where is_applied = 'yes' and tenant_id = :tenantId and status = 'draft'
            $sql = "update $tableName set is_active = '0'  where is_active = '1' and tenant_id = :tenantId and is_deleted = 0";
            $stmt = $pdo->prepare($sql);
            $stmt->bindParam(':tenantId', $tenantId);
            $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging\
            $stmt->execute();
            // Step 3: count how many rows are updated
            $rowCount = $stmt->rowCount();
            if ($rowCount === 0) {
                return [
                    'status' => 'Success',
                    'message' => 'No rows updated. None match requierments.'
                ];
            }else{
                // rows updated successfully message
                return [
                    'status' => 'success',
                    'message' => 'Rows updated successfully.',
                    'rowCount' => $rowCount
                ];
            }


            return [
                'status' => 'success',
                'message' => 'Rows found.',
                'rowCount' => $rowCount
            ];
        }
    }
    function updateWithChargeId($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId    = $inputData['foodOrder']['id'] ?? null;
        $chargeId   = $inputData['secure']['chargeId'] ?? null;
        $receiptUrl = $inputData['secure']['receiptUrl'] ?? null;

        if (!$orderId || !$chargeId) {
            return ['status' => 'failed', 'message' => 'Order ID and Charge ID are required'];
        }
        // Update the order with the charge ID
        $sql = "UPDATE food_order SET transaction_id = :charge_id, payment_status = 'paid', stripe_receipt_url = :receipt_url WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':charge_id', $chargeId);
        $stmt->bindValue(':order_id', $orderId);
        $stmt->bindValue(':receipt_url', $receiptUrl);
        try {
            $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging\
            $stmt->execute();
            return ['status' => 'success', 'message' => 'Order updated with charge ID'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
        
    }
    function updateCancelled($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId = $inputData['foodOrder']['id'] ?? null;

        if (!$orderId) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        // Update the order status to 'cancelled'
        $sql = "UPDATE food_order SET order_status = 'cancelled', payment_status = 'unpaid' WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':order_id', $orderId);
        try {
            $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging\
            $stmt->execute();
            return ['status' => 'success', 'message' => 'Order status updated to cancelled'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
    function updateSuccessful($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId = $inputData['foodOrder']['id'] ?? null;

        if (!$orderId) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        // Update the order status to 'confirmed'
        $sql = "UPDATE food_order SET order_status = 'confirmed' WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':order_id', $orderId);
        try {
            $stmt->execute();
            return ['status' => 'success', 'message' => 'Order status updated to confirmed'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
    function updateWithRefundId($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId  = $inputData['id'] ?? null;
        $refundId = $inputData['secure']['refundId'] ?? null;

        error_log("updateWithRefundId called with orderId: $orderId, refundId: $refundId");

        if (!$orderId ) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        if (!$refundId ) {
            return ['status' => 'failed', 'message' => 'Refund ID is required'];
        }

        $sql = "SELECT total_order_modified_reason, total_order_modified_price FROM food_order WHERE id = :order_id LIMIT 1";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':order_id', $orderId);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $currentReason = $result['total_order_modified_reason'] ?? '';
        $currentPrice = $result['total_order_modified_price'] ?? 0;
        $processRefund = $inputData['refundType'];

        if($processRefund == 'full'){
            $newCurrentPrice = 0;
            $refundStatus = 'refunded';
        }else if($processRefund == 'partial'){
            $refundAmount = $inputData['refundAmount'] ?? 0;
            $newCurrentPrice = $currentPrice - $refundAmount;
            $currentReason .= ($currentReason ? ', ' : '') . $processRefund . ' refund processed on ' . date('Y-m-d H:i:s') . ' original total modified price: ' . $currentPrice . ' new total modified price: ' . $newCurrentPrice;
            // if new current price is 0, set refund status to refunded
            if($newCurrentPrice == 0){
                $refundStatus = 'refunded';
            }else{
                $refundStatus = 'partially_refunded';
            }
        }else if($processRefund == 'itemBased'){
            $newCurrentPrice =  $currentPrice -($inputData['totalRefundAmount'] / 100) ?? 0;
            $currentReason .= ($currentReason ? ', ' : '') . $inputData['refundMessage'] ?? '';
            // if new current price is 0, set refund status to refunded
        }

        // Update the order with the refund ID


        $sql = "INSERT INTO food_order_history (tenant_id, order_id, change_type, change_data, table_of_source, changed_by) 
        VALUES (:tenant_id, :order_id, :change_type, :change_data, :table_of_source, :changed_by)";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':tenant_id', $inputData['tenantId']);
        $stmt->bindValue(':order_id', $orderId);
        $stmt->bindValue(':change_type', 'update');
        $stmt->bindValue(':change_data', json_encode(['refund_id' => $refundId, 'total_order_modified_reason' => $currentReason]));
        $stmt->bindValue(':table_of_source', 'food_order');
        $stmt->bindValue(':changed_by', $inputData['userId'] ?? 0); // Assuming userId is passed in inputData
        $stmt->execute();

        $sql = "UPDATE food_order SET refunded_id = :refund_id, payment_status = :refundStatus, total_order_modified_reason = :total_order_modified_reason, total_order_modified_price = :total_order_modified_price WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':refund_id', $refundId);
        $stmt->bindValue(':refundStatus', 'refunded');
        $stmt->bindValue(':total_order_modified_reason', $currentReason);
        $stmt->bindValue(':total_order_modified_price', $newCurrentPrice);
        $stmt->bindValue(':order_id', $orderId);
        try {
            $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging
            $stmt->execute();

            if($processRefund == 'full'){
                // If full refund, cancel all items in the order
                $inputData['id'] = $orderId;
                $res = cancelAllItemsInOrder($inputData);
                if($res['status'] == 'failed'){
                    return $res;
                }
            }

            if($processRefund == 'itemBased'){
                $items = $inputData['validatedItems'] ?? [];

                $tempInput = [];
                $tempInput['db']            = $inputData['db'];
                $tempInput['tenantId']      = $inputData['tenantId'];
                $tempInput['userId']        = $inputData['userId'] ?? 0;
                $tempInput['roleId']        = $inputData['roleId'] ?? 0;
                $tempInput['debug']         = $inputData['debug'] ?? null;
                $tempInput['secure']        = $inputData['secure'] ?? [];
                $tempInput['debugModeFlag'] = $inputData['debugModeFlag'] ?? null;

                foreach($items as $item){
                    $tempInput['foodOrderItem'] = [
                        'id'                    => $item['id'],
                        'itemStatus'            => $item['newModifiedPrice'] == 0 ? 'refunded' : 'partial_refund',
                        'itemModifiedPrice'     => $item['newModifiedPrice'] ?? null,
                        'itemModifiedReason'    => $item['itemModifiedReason'] ?? null,
                        'itemQuantity'          => $item['qty']
                    ];
                    $tempInput['recordType'] = 'update';
                    $tempInput['table'] = 'foodOrderItem';
                    $tempInput['dataKey'] = 'food_order_item';
                    
                    $res = updateContent($tempInput);
                    if($res['status'] == 'failed'){
                        return $res;
                    }
                }
            }


            return ['status' => 'success', 'message' => 'Order updated with refund ID'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
    function cancelAllItemsInOrder($inputData){
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId = $inputData['id'] ?? null;

        if (!$orderId) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        // Update the order items status to 'cancelled'
        $sql = "UPDATE food_order_item SET item_status = 'cancelled' WHERE order_id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':order_id', $orderId);
        try {
            $inputData['preparedStatement'] = $stmt; // Store the prepared statement for debugging\
            $stmt->execute();
            return ['status' => 'success', 'message' => 'All order items status updated to cancelled'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
?>