-- Cascading delete script for tenant cleanup
-- This script deletes all records related to a specific tenant
-- Working from leaf nodes to root to maintain referential integrity
-- 
-- Usage: Replace @tenant_id with the actual tenant ID to delete
-- Example: SET @tenant_id = 123;

SET @tenant_id = :tenant_id; -- Replace with actual tenant ID

-- Step 1: Delete from leaf tables (no other tables depend on these)
-- food_order_extra (depends on food_order_item)
DELETE food_order_extra 
FROM food_order_extra 
INNER JOIN food_order_item ON food_order_extra.item_id = food_order_item.id
INNER JOIN food_order ON food_order_item.order_id = food_order.id
WHERE food_order.tenant_id = @tenant_id;

-- Step 2: Delete order items (depends on food_order)
DELETE food_order_item 
FROM food_order_item 
INNER JOIN food_order ON food_order_item.order_id = food_order.id
WHERE food_order.tenant_id = @tenant_id;

-- Step 3: Delete order history (depends on food_order)
DELETE FROM food_order_history 
WHERE tenant_id = @tenant_id;

-- Step 4: Delete orders (this will help us track addresses and users to clean up)
-- First, collect address and guest user IDs that will be orphaned
CREATE TEMPORARY TABLE temp_addresses_to_check AS
SELECT DISTINCT delivery_address_id as address_id 
FROM food_order 
WHERE tenant_id = @tenant_id AND delivery_address_id IS NOT NULL
UNION
SELECT DISTINCT billing_address_id as address_id 
FROM food_order 
WHERE tenant_id = @tenant_id AND billing_address_id IS NOT NULL;

CREATE TEMPORARY TABLE temp_guest_users_to_check AS
SELECT DISTINCT guest_user_id 
FROM food_order 
WHERE tenant_id = @tenant_id AND guest_user_id IS NOT NULL;

-- Delete the orders
DELETE FROM food_order 
WHERE tenant_id = @tenant_id;

-- Step 5: Delete user address relationships for addresses that are no longer referenced
-- (only if the address is not used by other orders)
DELETE food_order_user_address 
FROM food_order_user_address 
INNER JOIN temp_addresses_to_check ON food_order_user_address.address_id = temp_addresses_to_check.address_id
WHERE NOT EXISTS (
    SELECT 1 FROM food_order 
    WHERE food_order.delivery_address_id = food_order_user_address.address_id 
       OR food_order.billing_address_id = food_order_user_address.address_id
);

-- Step 6: Delete tenant-specific address relationships
DELETE FROM food_address_to_tenant 
WHERE tenant_id = @tenant_id;

-- Step 7: Clean up orphaned addresses (not referenced by any orders)
DELETE food_order_address 
FROM food_order_address 
INNER JOIN temp_addresses_to_check ON food_order_address.id = temp_addresses_to_check.address_id
WHERE NOT EXISTS (
    SELECT 1 FROM food_order 
    WHERE food_order.delivery_address_id = food_order_address.id 
       OR food_order.billing_address_id = food_order_address.id
)
AND NOT EXISTS (
    SELECT 1 FROM food_order_user_address 
    WHERE food_order_user_address.address_id = food_order_address.id
)
AND NOT EXISTS (
    SELECT 1 FROM food_address_to_tenant 
    WHERE food_address_to_tenant.address_id = food_order_address.id
);

-- Step 8: Clean up orphaned guest users (not referenced by any orders)
DELETE food_order_guest_user 
FROM food_order_guest_user 
INNER JOIN temp_guest_users_to_check ON food_order_guest_user.id = temp_guest_users_to_check.guest_user_id
WHERE NOT EXISTS (
    SELECT 1 FROM food_order 
    WHERE food_order.guest_user_id = food_order_guest_user.id
);

-- Clean up temporary tables
DROP TEMPORARY TABLE temp_addresses_to_check;
DROP TEMPORARY TABLE temp_guest_users_to_check;

-- Verification queries (optional - uncomment to verify deletion)
/*
SELECT 'Orders remaining for tenant' as check_type, COUNT(*) as count 
FROM food_order WHERE tenant_id = @tenant_id
UNION ALL
SELECT 'Order items remaining' as check_type, COUNT(*) as count 
FROM food_order_item foi 
INNER JOIN food_order fo ON foi.order_id = fo.id 
WHERE fo.tenant_id = @tenant_id
UNION ALL
SELECT 'Order extras remaining' as check_type, COUNT(*) as count 
FROM food_order_extra foe 
INNER JOIN food_order_item foi ON foe.item_id = foi.id 
INNER JOIN food_order fo ON foi.order_id = fo.id 
WHERE fo.tenant_id = @tenant_id
UNION ALL
SELECT 'Order history remaining' as check_type, COUNT(*) as count 
FROM food_order_history WHERE tenant_id = @tenant_id
UNION ALL
SELECT 'Address relationships remaining' as check_type, COUNT(*) as count 
FROM food_address_to_tenant WHERE tenant_id = @tenant_id;
*/