-- Cascading delete script for food-menu tenant cleanup
-- This script deletes all menu records related to a specific tenant
-- Working from leaf nodes to root to maintain referential integrity
-- 
-- Usage: Replace @tenant_id with the actual tenant ID to delete
-- Example: SET @tenant_id = 123;

SET @tenant_id = :tenant_id; -- Replace with actual tenant ID

-- Step 1: Delete from deepest leaf tables first
-- menu_meal_part_items (depends on menu_meal_parts and menu_items)
DELETE menu_meal_part_items 
FROM menu_meal_part_items 
INNER JOIN menu_meal_parts ON menu_meal_part_items.part_id = menu_meal_parts.id
INNER JOIN menu_meals ON menu_meal_parts.meal_id = menu_meals.id
INNER JOIN menu_categories ON menu_meals.category_id = menu_categories.id
INNER JOIN menu_menus ON menu_categories.menu_id = menu_menus.id
WHERE menu_menus.tenant_id = @tenant_id;

-- Step 2: Delete meal parts (depends on menu_meals)
DELETE menu_meal_parts 
FROM menu_meal_parts 
INNER JOIN menu_meals ON menu_meal_parts.meal_id = menu_meals.id
INNER JOIN menu_categories ON menu_meals.category_id = menu_categories.id
INNER JOIN menu_menus ON menu_categories.menu_id = menu_menus.id
WHERE menu_menus.tenant_id = @tenant_id;

-- Step 3: Delete meals (depends on menu_categories)
DELETE menu_meals 
FROM menu_meals 
INNER JOIN menu_categories ON menu_meals.category_id = menu_categories.id
INNER JOIN menu_menus ON menu_categories.menu_id = menu_menus.id
WHERE menu_menus.tenant_id = @tenant_id;

-- Step 4: Delete option joins (depends on menu_option_groups and menu_option_extra)
DELETE FROM menu_option_joins 
WHERE group_id IN (
    SELECT id FROM menu_option_groups WHERE tenant_id = @tenant_id
);

-- Step 5: Delete item to option links (depends on menu_items and menu_option_groups)
DELETE menu_item_to_options 
FROM menu_item_to_options 
INNER JOIN menu_option_groups ON menu_item_to_options.group_id = menu_option_groups.id
WHERE menu_option_groups.tenant_id = @tenant_id;

-- Also delete item to option links by item
DELETE menu_item_to_options 
FROM menu_item_to_options 
INNER JOIN menu_items ON menu_item_to_options.item_id = menu_items.id
WHERE menu_items.tenant_id = @tenant_id;

-- Step 6: Delete category-item links (depends on menu_categories and menu_items)
DELETE menu_category_items 
FROM menu_category_items 
INNER JOIN menu_categories ON menu_category_items.category_id = menu_categories.id
INNER JOIN menu_menus ON menu_categories.menu_id = menu_menus.id
WHERE menu_menus.tenant_id = @tenant_id;

-- Step 7: Collect nutrition profiles that will be orphaned for cleanup
CREATE TEMPORARY TABLE temp_nutrition_to_check AS
SELECT DISTINCT nutrition_id 
FROM menu_items 
WHERE tenant_id = @tenant_id AND nutrition_id IS NOT NULL
UNION
SELECT DISTINCT nutrition_id 
FROM menu_option_extra 
WHERE tenant_id = @tenant_id AND nutrition_id IS NOT NULL;

-- Step 8: Delete menu items (depends on menu_nutrition_profiles)
DELETE FROM menu_items 
WHERE tenant_id = @tenant_id;

-- Step 9: Delete option extras (depends on menu_nutrition_profiles)
DELETE FROM menu_option_extra 
WHERE tenant_id = @tenant_id;

-- Step 10: Delete option groups
DELETE FROM menu_option_groups 
WHERE tenant_id = @tenant_id;

-- Step 11: Delete categories (depends on menu_menus, has self-reference for parent_id)
-- Delete child categories first, then parent categories
DELETE FROM menu_categories 
WHERE menu_id IN (SELECT id FROM menu_menus WHERE tenant_id = @tenant_id)
  AND parent_id IS NOT NULL;

-- Delete parent categories
DELETE FROM menu_categories 
WHERE menu_id IN (SELECT id FROM menu_menus WHERE tenant_id = @tenant_id);

-- Step 12: Delete menus (root table for this tenant)
DELETE FROM menu_menus 
WHERE tenant_id = @tenant_id;

-- Step 13: Clean up orphaned nutrition profiles (only if not referenced by other tenants)
DELETE menu_nutrition_profiles 
FROM menu_nutrition_profiles 
INNER JOIN temp_nutrition_to_check ON menu_nutrition_profiles.id = temp_nutrition_to_check.nutrition_id
WHERE NOT EXISTS (
    SELECT 1 FROM menu_items 
    WHERE menu_items.nutrition_id = menu_nutrition_profiles.id
)
AND NOT EXISTS (
    SELECT 1 FROM menu_option_extra 
    WHERE menu_option_extra.nutrition_id = menu_nutrition_profiles.id
);

-- Step 14: Clean up audit logs for this tenant
DELETE FROM menu_audit_log 
WHERE tenant_id = @tenant_id;

-- Clean up temporary table
DROP TEMPORARY TABLE temp_nutrition_to_check;

-- Verification queries (optional - uncomment to verify deletion)
/*
SELECT 'Menus remaining for tenant' as check_type, COUNT(*) as count 
FROM menu_menus WHERE tenant_id = @tenant_id
UNION ALL
SELECT 'Categories remaining' as check_type, COUNT(*) as count 
FROM menu_categories mc 
INNER JOIN menu_menus mm ON mc.menu_id = mm.id 
WHERE mm.tenant_id = @tenant_id
UNION ALL
SELECT 'Items remaining' as check_type, COUNT(*) as count 
FROM menu_items WHERE tenant_id = @tenant_id
UNION ALL
SELECT 'Option groups remaining' as check_type, COUNT(*) as count 
FROM menu_option_groups WHERE tenant_id = @tenant_id
UNION ALL
SELECT 'Option extras remaining' as check_type, COUNT(*) as count 
FROM menu_option_extra WHERE tenant_id = @tenant_id
UNION ALL
SELECT 'Option joins remaining' as check_type, COUNT(*) as count 
FROM menu_option_joins moj 
INNER JOIN menu_option_groups mog ON moj.group_id = mog.id 
WHERE mog.tenant_id = @tenant_id
UNION ALL
SELECT 'Item to options remaining' as check_type, COUNT(*) as count 
FROM menu_item_to_options mito 
INNER JOIN menu_option_groups mog ON mito.group_id = mog.id 
WHERE mog.tenant_id = @tenant_id
UNION ALL
SELECT 'Category items remaining' as check_type, COUNT(*) as count 
FROM menu_category_items mci 
INNER JOIN menu_categories mc ON mci.category_id = mc.id 
INNER JOIN menu_menus mm ON mc.menu_id = mm.id 
WHERE mm.tenant_id = @tenant_id
UNION ALL
SELECT 'Meals remaining' as check_type, COUNT(*) as count 
FROM menu_meals mm2 
INNER JOIN menu_categories mc ON mm2.category_id = mc.id 
INNER JOIN menu_menus mm ON mc.menu_id = mm.id 
WHERE mm.tenant_id = @tenant_id
UNION ALL
SELECT 'Meal parts remaining' as check_type, COUNT(*) as count 
FROM menu_meal_parts mmp 
INNER JOIN menu_meals mm2 ON mmp.meal_id = mm2.id 
INNER JOIN menu_categories mc ON mm2.category_id = mc.id 
INNER JOIN menu_menus mm ON mc.menu_id = mm.id 
WHERE mm.tenant_id = @tenant_id
UNION ALL
SELECT 'Audit logs remaining' as check_type, COUNT(*) as count 
FROM menu_audit_log WHERE tenant_id = @tenant_id;
*/