-- Full menu schema and audit triggers (with  wrapping)


-- Menu Table
CREATE TABLE menu_menus (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    menu_name TEXT NOT NULL,
    menu_description TEXT,
    is_active INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

insert into menu_menus (id, tenant_id, `menu_name`, `menu_description`, is_active, is_deleted) 
values 
(1, 1, 'Test Menu', 'Test Description', 0, 1);

-- Categories
CREATE TABLE menu_categories (
    id INTEGER PRIMARY KEY,
    menu_id INTEGER NOT NULL,
    parent_id INTEGER,
    category_name TEXT NOT NULL,
    category_description TEXT,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (menu_id) REFERENCES menu_menus(id),
    FOREIGN KEY (parent_id) REFERENCES menu_categories(id)
);

-- Items
CREATE TABLE menu_items (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    item_name TEXT NOT NULL,
    item_description TEXT,
    base_price REAL,
    nutrition_id INTEGER,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)
);

-- Category-Item Linking
CREATE TABLE menu_category_items (
    id INTEGER PRIMARY KEY,
    category_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- Nutrition Profiles
CREATE TABLE menu_nutrition_profiles (
    id INTEGER PRIMARY KEY,
    calories INTEGER DEFAULT 0,
    protein_grams REAL DEFAULT 0,
    fat_grams REAL DEFAULT 0,
    carbs_grams REAL DEFAULT 0,
    sugar_grams REAL DEFAULT 0,
    salt_grams REAL DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Option Groups
CREATE TABLE menu_option_groups (
    id INTEGER PRIMARY KEY,
    group_name TEXT NOT NULL,
    group_description TEXT,
    tenant_id INTEGER NOT NULL,
    min_select INTEGER DEFAULT 0,
    max_select INTEGER DEFAULT 1,
    is_required INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- option Joining Groups
CREATE TABLE menu_item_to_options (
    id                  INTEGER PRIMARY KEY,    
    group_id            INTEGER NOT NULL,
    item_id             INTEGER NOT NULL,
    is_active           INTEGER DEFAULT 0,
    is_deleted          INTEGER DEFAULT 0,
    created_at          DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at          DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (group_id) REFERENCES menu_option_groups(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- Option Items
CREATE TABLE menu_option_extra (
    id                       INTEGER PRIMARY KEY,
    tenant_id                TEXT NOT NULL,
    extra_name               TEXT,
    extra_description        REAL DEFAULT 0,
    price_adjustment         INTEGER,
    nutrition_id             INTEGER DEFAULT 0,
    is_active                INTEGER DEFAULT 0,
    is_deleted               INTEGER DEFAULT 0,
    created_at               DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at               DATETIME DEFAULT CURRENT_TIMESTAMP
);
--  FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)

create table menu_option_joins (
    id                 INTEGER PRIMARY KEY,
    group_id           INTEGER NOT NULL,
    extra_id           INTEGER NOT NULL,
    price_adjustment   REAL DEFAULT 0,
    is_default         INTEGER DEFAULT 0,
    is_active          INTEGER DEFAULT 0,
    is_deleted         INTEGER DEFAULT 0,
    created_at         DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at         DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (group_id) REFERENCES menu_option_groups(id),
    FOREIGN KEY (extra_id) REFERENCES menu_option_extra(id)
);

-- Meals
CREATE TABLE menu_meals (
    id INTEGER PRIMARY KEY,
    category_id INTEGER NOT NULL,
    meal_name TEXT NOT NULL,
    meal_description TEXT,
    base_price REAL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id)
);

-- Meal Parts
CREATE TABLE menu_meal_parts (
    id INTEGER PRIMARY KEY,
    meal_id INTEGER NOT NULL,
    meal_part_name TEXT NOT NULL,
    meal_part_description TEXT,
    min_select INTEGER DEFAULT 1,
    max_select INTEGER DEFAULT 1,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (meal_id) REFERENCES menu_meals(id)
);

-- Meal Part Items
CREATE TABLE menu_meal_part_items (
    part_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (part_id, item_id),
    FOREIGN KEY (part_id) REFERENCES menu_meal_parts(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- =====================
-- AUDIT LOG TABLE SETUP
-- =====================
CREATE TABLE menu_audit_log (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    table_of_source TEXT NOT NULL,
    record_id INTEGER NULL,
    action_type TEXT NOT NULL,
    old_value TEXT,
    action_by INTEGER DEFAULT 0,
    action_at DATETIME DEFAULT CURRENT_TIMESTAMP
);


-- Ensure only one active menu per tenant on INSERT
CREATE TRIGGER trg_only_one_active_menu
BEFORE INSERT ON menu_menus
FOR EACH ROW
WHEN NEW.is_active = 1
BEGIN
    UPDATE menu_menus
    SET is_active = 0
    WHERE tenant_id = NEW.tenant_id AND is_active = 1 AND is_deleted = 0;
END;

-- Ensure only one active menu per tenant on UPDATE
CREATE TRIGGER trg_only_one_active_menu_update
BEFORE UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.is_active = 1 AND OLD.is_active != 1
BEGIN
    UPDATE menu_menus
    SET is_active = 0
    WHERE tenant_id = NEW.tenant_id AND id != NEW.id AND is_active = 1 AND is_deleted = 0;
END;


-- =============================
-- menu_items: UPDATE
-- =============================
CREATE TRIGGER trg_menu_items_update
AFTER UPDATE ON menu_items
FOR EACH ROW
WHEN NEW.item_name != OLD.item_name OR NEW.base_price != OLD.base_price OR NEW.is_deleted != OLD.is_deleted
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT m.tenant_id FROM menu_category_items ci
         JOIN menu_categories c ON ci.category_id = c.id
         JOIN menu_menus m ON c.menu_id = m.id
         WHERE ci.item_id = NEW.id LIMIT 1),
        'menu_items',
        NEW.id,
        'updated',
        'changed item fields',
        0
    );
END;

-- =============================
-- menu_items: DELETE
-- =============================
CREATE TRIGGER trg_menu_items_delete
BEFORE DELETE ON menu_items
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT m.tenant_id FROM menu_category_items ci
         JOIN menu_categories c ON ci.category_id = c.id
         JOIN menu_menus m ON c.menu_id = m.id
         WHERE ci.item_id = OLD.id LIMIT 1),
        'menu_items',
        OLD.id,
        'deleted',
        'deleted item',
        0
    );
END;

-- =============================
-- menu_menus: UPDATE
-- =============================
CREATE TRIGGER trg_menu_menus_update
AFTER UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.menu_name != OLD.menu_name OR NEW.is_active != OLD.is_active OR NEW.is_deleted != OLD.is_deleted
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        NEW.tenant_id,
        'menu_menus',
        NEW.id,
        'updated',
        'changed menu fields',
        0
    );
END;

-- =============================
-- menu_menus: DELETE
-- =============================
CREATE TRIGGER trg_menu_menus_delete
BEFORE DELETE ON menu_menus
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        OLD.tenant_id,
        'menu_menus',
        OLD.id,
        'deleted',
        'deleted menu',
        0
    );
END;

-- =============================
-- menu_categories: UPDATE
-- =============================
CREATE TRIGGER trg_menu_categories_update
AFTER UPDATE ON menu_categories
FOR EACH ROW
WHEN NEW.category_name != OLD.category_name OR NEW.is_deleted != OLD.is_deleted
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT tenant_id FROM menu_menus WHERE id = NEW.menu_id),
        'menu_categories',
        NEW.id,
        'updated',
        'changed category fields',
        0
    );
END;

-- =============================
-- menu_categories: DELETE
-- =============================
CREATE TRIGGER trg_menu_categories_delete
BEFORE DELETE ON menu_categories
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT tenant_id FROM menu_menus WHERE id = OLD.menu_id),
        'menu_categories',
        OLD.id,
        'deleted',
        'deleted category',
        0
    );
END;

-- =============================
-- menu_meals: UPDATE
-- =============================
CREATE TRIGGER trg_menu_meals_update
AFTER UPDATE ON menu_meals
FOR EACH ROW
WHEN NEW.meal_name != OLD.meal_name OR NEW.base_price != OLD.base_price OR NEW.is_deleted != OLD.is_deleted
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT tenant_id FROM menu_menus WHERE id = (SELECT menu_id FROM menu_categories WHERE id = NEW.category_id)),
        'menu_meals',
        NEW.id,
        'updated',
        'changed meal fields',
        0
    );
END;

-- =============================
-- menu_meals: DELETE
-- =============================
CREATE TRIGGER trg_menu_meals_delete
BEFORE DELETE ON menu_meals
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT tenant_id FROM menu_menus WHERE id = (SELECT menu_id FROM menu_categories WHERE id = OLD.category_id)),
        'menu_meals',
        OLD.id,
        'deleted',
        'deleted meal',
        0
    );
END;


-- =============================
-- menu_option_extra: DELETE
-- =============================
CREATE TRIGGER trg_menu_option_extra_delete
BEFORE DELETE ON menu_option_extra
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    )
    VALUES (
        (SELECT m.tenant_id
         FROM menu_option_groups g
         JOIN menu_items i ON g.item_id = i.id
         JOIN menu_category_items ci ON ci.item_id = i.id
         JOIN menu_categories c ON ci.category_id = c.id
         JOIN menu_menus m ON c.menu_id = m.id
         WHERE g.id = OLD.group_id LIMIT 1),
        'menu_option_extra',
        OLD.id,
        'deleted',
        'deleted option item',
        0
    );
END;

-- Trigger: Update is_deleted status in menu_categories when menu_menus is updated
CREATE TRIGGER trg_menu_menus_is_deleted
AFTER UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.is_deleted != OLD.is_deleted
BEGIN
    UPDATE menu_categories
    SET is_deleted = NEW.is_deleted
    WHERE menu_id = NEW.id;
END;

-- Trigger: Update is_deleted status in menu_items when menu_categories is updated
CREATE TRIGGER trg_mark_category_items_deleted
AFTER UPDATE OF is_deleted ON menu_items
FOR EACH ROW
WHEN NEW.is_deleted = 1 AND OLD.is_deleted = 0
BEGIN
    UPDATE menu_category_items
    SET is_deleted = 1,
        updated_at = CURRENT_TIMESTAMP
    WHERE item_id = NEW.id;
END;


CREATE TRIGGER trg_option_group_disable_or_delete_joins
AFTER UPDATE OF is_active, is_deleted ON menu_option_groups
FOR EACH ROW
WHEN NEW.is_deleted = 1 OR NEW.is_active = 0
BEGIN
    UPDATE menu_option_joins
    SET 
        is_deleted = NEW.is_deleted,
        is_active = NEW.is_active,
        updated_at = CURRENT_TIMESTAMP
    WHERE group_id = NEW.id;
END;

CREATE TRIGGER trg_option_group_cascade_items
AFTER UPDATE OF is_active, is_deleted ON menu_option_groups
FOR EACH ROW
WHEN NEW.is_deleted = 1 OR NEW.is_active = 0
BEGIN
    UPDATE menu_option_extra
    SET 
        is_deleted = NEW.is_deleted,
        is_active = NEW.is_active,
        updated_at = CURRENT_TIMESTAMP
    WHERE id IN (
        SELECT j.extra_id
        FROM menu_option_joins j
        WHERE j.group_id = NEW.id
        AND NOT EXISTS (
            SELECT 1
            FROM menu_option_joins j2
            WHERE j2.extra_id = j.extra_id
              AND j2.group_id != NEW.id
              AND j2.is_deleted = 0
        )
    );
END;

DROP TRIGGER IF EXISTS trg_option_extra_status_to_joins;
CREATE TRIGGER trg_option_extra_status_to_joins
AFTER UPDATE OF is_active, is_deleted ON menu_option_extra
FOR EACH ROW
WHEN NEW.is_active = 0 OR NEW.is_deleted = 1
BEGIN
    UPDATE menu_option_joins
    SET 
        is_active = NEW.is_active,
        is_deleted = NEW.is_deleted,
        updated_at = CURRENT_TIMESTAMP
    WHERE extra_id = NEW.id;
END;