<?php
// Orbit Unit Test Runner - PHP Wrapper
// ===================================
//
// Cross-platform test runner for Orbit tests.
// - Runs from script directory
// - Ensures reports/ exists
// - Runs runTest.php if present
// - Logs output to reports/ with timestamp
// - Echoes output to console AND writes to log
// - Exits with the same exit code as the test run

declare(strict_types=1);

function isTty(): bool {
    // Best-effort; fine if it returns false in Jenkins
    if (PHP_OS_FAMILY === 'Windows') {
        return function_exists('sapi_windows_vt100_support') ? true : false;
    }
    return function_exists('posix_isatty') ? @posix_isatty(STDOUT) : false;
}

function color(string $text, string $ansiCode): string {
    static $useColor = null;
    if ($useColor === null) $useColor = isTty();
    return $useColor ? "\033[{$ansiCode}m{$text}\033[0m" : $text;
}

function println(string $line = ''): void {
    echo $line . PHP_EOL;
}

function ensureDir(string $path): void {
    if (!is_dir($path)) {
        if (!@mkdir($path, 0777, true) && !is_dir($path)) {
            throw new RuntimeException("Failed to create directory: {$path}");
        }
    }
}

function timestamp(): string {
    return date('Y-m-d_H-i-s');
}

function runPhpAndTee(string $phpFile, string $logFile): int {
    $phpBin = PHP_BINARY;

    // Merge stderr into stdout: 2>&1
    $cmd = escapeshellarg($phpBin) . ' ' . escapeshellarg($phpFile) . ' 2>&1';

    $descriptors = [
        0 => ["pipe", "r"],
        1 => ["pipe", "w"],
        2 => ["pipe", "w"], // still defined, but stderr is merged so should be quiet
    ];

    $proc = proc_open($cmd, $descriptors, $pipes, getcwd());

    if (!is_resource($proc)) {
        throw new RuntimeException("Failed to start process: {$cmd}");
    }

    fclose($pipes[0]);

    $logHandle = fopen($logFile, 'ab');
    if ($logHandle === false) $logHandle = null;

    // IMPORTANT: blocking mode
    stream_set_blocking($pipes[1], true);

    while (!feof($pipes[1])) {
        $chunk = fread($pipes[1], 8192);
        if ($chunk === '' || $chunk === false) {
            usleep(20000);
            continue;
        }
        echo $chunk;
        if ($logHandle) fwrite($logHandle, $chunk);
        flush();
        if (function_exists('ob_flush')) @ob_flush();
    }

    fclose($pipes[1]);
    if (isset($pipes[2]) && is_resource($pipes[2])) fclose($pipes[2]);
    if ($logHandle) fclose($logHandle);

    $exitCode = proc_close($proc);
    return is_int($exitCode) ? $exitCode : 1;
}



// ------------------------------------------------------------

try {
    // Clear screen (best effort)
    if (PHP_OS_FAMILY !== 'Windows') {
        @system('clear');
    } else {
        @system('cls');
    }

    // Change to script directory (project root)
    $scriptDir = __DIR__;
    chdir($scriptDir);

    println(color("Working from: " . getcwd(), "36"));
    println(color(str_repeat("=", 63), "33"));
    println(color("Orbit Unit Test Suite - Running All Tests", "33"));
    println(color(str_repeat("=", 63), "33"));

    $exitCode = 0;

    if (file_exists("runTest.php")) {
        println();
        println(color("Running Waitron Tests...", "32"));
        println(color("-------------------------------------", "90"));

        ensureDir("reports");
        $logFile = "reports/shop-test-output_" . timestamp() . ".log";

        $lastExit = runPhpAndTee("runTest.php", $logFile);

        if ($lastExit !== 0) {
            $exitCode = $lastExit;
            println(color("Shop tests failed with exit code: {$lastExit}", "31"));
            println(color("Shop test log saved to: {$logFile}", "36"));
        } else {
            println(color("Waitron-Shop tests completed successfully", "32"));
            println(color("Shop test log saved to: {$logFile}", "36"));
        }
    } else {
        println(color("Waitron test runner not found", "33"));
    }

    println();
    println(color(str_repeat("=", 63), "33"));
    if ($exitCode === 0) {
        println(color("All tests completed successfully!", "32"));
    } else {
        println(color("Some tests failed. Check output above for details.", "31"));
    }
    println(color(str_repeat("=", 63), "33"));

    exit($exitCode);

} catch (Throwable $e) {
    // Hard fail if the runner itself breaks
    fwrite(STDERR, "Runner error: " . $e->getMessage() . PHP_EOL);
    exit(1);
}
