<?php
/**
 * Test Bootstrap (UTC + Deterministic "Now")
 * ------------------------------------------
 * - Forces UTC for PHP and (optionally) DB sessions.
 * - Uses a shared SQLite file DB across all tests.
 * - Loads package schemas once.
 * - Exposes a single universal "now" (UTC) via $GLOBALS['TEST_NOW_UTC'].
 * - Provides helpers for cutoffs and quick debugging.
 *
 * ENV overrides:
 *   TEST_NOW_UTC="2025-08-25 17:39:20"  (ISO UTC string) to fix the test clock.
 */

// -------------------------------------
// 0) Output buffering + session safety
// -------------------------------------
ob_start();

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// -------------------------------------
// 1) Composer + basic env flags
// -------------------------------------
require_once __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'vendor' . DIRECTORY_SEPARATOR . 'autoload.php';

if (!isset($_SERVER['REQUEST_METHOD'])) {
    $_SERVER['REQUEST_METHOD'] = 'CLI';
}

if (!defined('TEST_ENV')) {
    define('TEST_ENV', true);
}

// -------------------------------------
// 2) Universal time: force PHP to UTC
//    and define a deterministic "now"
// -------------------------------------
date_default_timezone_set('UTC');

$fixedNow = getenv('TEST_NOW_UTC'); // e.g. "2025-08-25 17:39:20"
if ($fixedNow) {
    // Validate simple format (best-effort)
    if (!preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/', $fixedNow)) {
        fwrite(STDERR, "⚠️  TEST_NOW_UTC is not 'YYYY-MM-DD HH:MM:SS'. Ignoring.\n");
        $fixedNow = null;
    }
}
$GLOBALS['TEST_NOW_UTC'] = $fixedNow ?: gmdate('Y-m-d H:i:s'); // default to current UTC

/**
 * Returns the fixed test "now" as \DateTimeImmutable (UTC).
 */
function testNowUtc(): \DateTimeImmutable {
    return new \DateTimeImmutable($GLOBALS['TEST_NOW_UTC'] . ' UTC');
}

/**
 * Returns a UTC cutoff string 'YYYY-MM-DD HH:MM:SS' equal to (test now - $minutes).
 */
function utcCutoffMinutes(int $minutes): string {
    return testNowUtc()
        ->sub(new DateInterval('PT' . max(0, $minutes) . 'M'))
        ->format('Y-m-d H:i:s');
}

/**
 * (Optional) Pin a PDO connection's session to UTC where supported.
 * No-op for SQLite (it's UTC for datetime('now')).
 */
// Either in tests bootstrap OR in generic.php (pick one place)

if (!function_exists('pinConnectionTimezoneToUtc')) {
    function pinConnectionTimezoneToUtc(PDO $pdo): void {
        $driver = $pdo->getAttribute(\PDO::ATTR_DRIVER_NAME);
        if ($driver === 'mysql') {
            $pdo->exec("SET time_zone = '+00:00'");
        } elseif ($driver === 'pgsql') {
            $pdo->exec("SET TIME ZONE 'UTC'");
        }
        // SQLite: no-op
    }
}


// -------------------------------------
// 3) Shared SQLite file DB for tests
// -------------------------------------
define('TEST_DB_PATH', sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'phpunit_test.sqlite');

// If running in CI (Jenkins or a workspace present), ensure the DB is fresh
// for this build by removing any pre-existing SQLite file and its WAL/SHM
// sidecars before creating the PDO connection. Local dev runs without
// CI env vars will keep the current behavior.
$ciPresent = (getenv('JENKINS_HOME') !== false) || (getenv('BUILD_NUMBER') !== false) || (getenv('WORKSPACE') !== false);
if ($ciPresent) {
    $toRemove = [TEST_DB_PATH, TEST_DB_PATH . '-wal', TEST_DB_PATH . '-shm'];
    foreach ($toRemove as $file) {
        if (file_exists($file)) {
            @unlink($file);
        }
    }
}

if (!isset($GLOBALS['mockPdo'])) {
    $GLOBALS['mockPdo'] = new PDO('sqlite:' . TEST_DB_PATH, null, null, [
        \PDO::ATTR_ERRMODE            => \PDO::ERRMODE_EXCEPTION,
        \PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_ASSOC,
        \PDO::ATTR_EMULATE_PREPARES   => false,
    ]);
    // SQLite pragmas for correctness + perf
    $GLOBALS['mockPdo']->exec('PRAGMA foreign_keys = ON;');
    $GLOBALS['mockPdo']->exec('PRAGMA journal_mode = WAL;');
    $GLOBALS['mockPdo']->exec('PRAGMA synchronous = NORMAL;');
    $GLOBALS['mockPdo']->exec('PRAGMA busy_timeout = 2000;');

    // Not strictly needed for SQLite, but harmless:
    pinConnectionTimezoneToUtc($GLOBALS['mockPdo']);

    // -------------------------------------
    // 4) Load all schemas from module test directories in numerical order (ONCE only)
    // -------------------------------------
    $projectRoot = dirname(__DIR__, 3); // Go up from core/unit-test/scripts to project root
    $allSchemas = [];
    
    if (!isset($GLOBALS['loadedSchemas'])) {
        $GLOBALS['loadedSchemas'] = [];
    }
    
    // Check if core table exists - if so, schemas are already loaded
    $tablesExist = false;
    try {
        $result = $GLOBALS['mockPdo']->query("SELECT name FROM sqlite_master WHERE type='table' AND name='request_log'");
        $tablesExist = $result && $result->fetchColumn() !== false;
    } catch (\PDOException $e) {
        $tablesExist = false;
    }
    
    if ($tablesExist) {
        echo "\n✅ Database tables already exist, skipping schema loading";
        echo "\n🔄 Using existing database structure";
    } else {
        echo "\n🏗️ First run - creating database schema...";
        
        // Scan all module directories for test SQL files
        $moduleDirs = glob($projectRoot . '/*', GLOB_ONLYDIR) ?: [];
        foreach ($moduleDirs as $moduleDir) {
            $moduleName = basename($moduleDir);
            
            // Skip non-module directories
            if (in_array($moduleName, ['vendor', 'tests', 'scripts', '.git', '.phpunit.cache'], true)) {
                continue;
            }
            
            $moduleTestsDir = $moduleDir . '/tests/';
            if (is_dir($moduleTestsDir)) {
                $sqlFiles = glob($moduleTestsDir . '*.sql') ?: [];
                foreach ($sqlFiles as $sqlFile) {
                    $filename = basename($sqlFile);
                    // Extract number prefix (e.g., "03" from "03-login.test.sql")
                    if (preg_match('/^(\d+)-/', $filename, $matches)) {
                        $order = (int)$matches[1];
                        $allSchemas[$order] = [
                            'file' => $sqlFile,
                            'name' => $filename,
                            'module' => $moduleName
                        ];
                    }
                }
            }
        }
        
        // Sort schemas by numerical order and load them
        ksort($allSchemas);
        
        $loadedCount = 0;
        foreach ($allSchemas as $order => $schema) {
            $schemaName = $schema['name'];
            if (!in_array($schemaName, $GLOBALS['loadedSchemas'], true)) {
                $sql = file_get_contents($schema['file']);
                if ($sql !== false && trim($sql) !== '') {
                    try {
                        $GLOBALS['mockPdo']->exec($sql);
                        $GLOBALS['loadedSchemas'][] = $schemaName;
                        $loadedCount++;
                        echo "\n✅ Loaded schema: {$schemaName} (from {$schema['module']})";
                    } catch (\PDOException $e) {
                        echo "\n❌ Failed to load schema {$schemaName}: " . $e->getMessage();
                        // Continue with other schemas
                    }
                }
            }
        }
        
        if ($loadedCount > 0) {
            echo "\n✅ Database schema created successfully with {$loadedCount} modules.";
            echo "\n📊 Ready for testing with: " . implode(', ', $GLOBALS['loadedSchemas']);
        } else {
            echo "\n⚠️ No numbered SQL schema files found in module test directories";
        }
    }
}

// -------------------------------------
// 5) Global inputData used by your code
// -------------------------------------
if (!isset($GLOBALS['inputData'])) {
    $GLOBALS['inputData'] = [
        'db' => [
            'dbApp'   => $GLOBALS['mockPdo'],
            'dbLogin' => $GLOBALS['mockPdo'],
        ],
        'secure' => [
            'stripeAccountKey' => 'test_stripe_account_key',
            'stripeSecretKey'  => 'test_stripe_secret_key',
        ]
    ];
}

// -------------------------------------
// 6) Env flags for app code
// -------------------------------------
$_ENV['DB_CONNECTION'] = 'sqlite';
putenv('DB_CONNECTION=sqlite');

// -------------------------------------
// 7) Errors + logging
// -------------------------------------
error_reporting(E_ALL);
ini_set('display_errors', '1');
ini_set('log_errors', '1');

set_error_handler(function ($errno, $errstr, $errfile, $errline) {
    // Echo to stdout for fast feedback in CI
    echo "[PHP {$errno}] {$errstr} in {$errfile} on line {$errline}\n";
    return false; // continue default handling too
});

// -------------------------------------
// 8) Optional: convenience seed helpers
// -------------------------------------

/**
 * Example seeding helper for orders relative to TEST_NOW_UTC.
 * Call in your tests when you need data:
 *
 *   seedOrders($GLOBALS['mockPdo'], [
 *     ['status' => 'pending', 'pay' => 'payment_pre_auth', 'ageMin' => 121],
 *     ['status' => 'pending', 'pay' => 'payment_pre_auth', 'ageMin' => 120],
 *     ['status' => 'pending', 'pay' => 'payment_pre_auth', 'ageMin' => 75],
 *     ['status' => 'completed', 'pay' => 'paid', 'ageMin' => 10],
 *   ]);
 */
function seedOrders(PDO $db, array $rows): void {
    $sql = <<<SQL
INSERT INTO food_order
(guest_user_id, tenant_id, delivery_address_id, billing_address_id,
 order_status, order_type, total_order_price, total_order_modified_price,
 total_order_modified_reason, payment_status, comment, tenant_comment,
 currency, transaction_id, payment_intent_id, created_at)
VALUES
 (NULL, 1, 1, 1, :status, 'delivery', 14.30, 1.30, 'Seeded in test', :pay, NULL, NULL, 'GBP', NULL, 'pi_seed', datetime(:now, :offset));
SQL;
    $stmt = $db->prepare($sql);
    foreach ($rows as $r) {
        $status  = $r['status']  ?? 'pending';
        $pay     = $r['pay']     ?? 'payment_pre_auth';
        $ageMin  = (int)($r['ageMin'] ?? 0); // minutes old
        $offset  = sprintf('-%d minutes', $ageMin);
        $stmt->execute([
            ':status' => $status,
            ':pay'    => $pay,
            ':now'    => $GLOBALS['TEST_NOW_UTC'],
            ':offset' => $offset,
        ]);
    }
}

/**
 * Debug a single row's aging decision quickly (SQLite).
 */
function debugAgeCheck(PDO $db, int $id, int $thresholdMin = 120): array {
    $cutoff = utcCutoffMinutes($thresholdMin);
    $stmt = $db->prepare("
        SELECT id, created_at,
               :now AS now_utc,
               :cut AS cutoff_utc,
               CAST((julianday(:now) - julianday(created_at)) * 24 * 60 AS INTEGER) AS age_min,
               (created_at <= :cut) AS matches
        FROM food_order
        WHERE id = :id
    ");
    $stmt->execute([
        ':now' => $GLOBALS['TEST_NOW_UTC'],
        ':cut' => $cutoff,
        ':id'  => $id,
    ]);
    $row = $stmt->fetch() ?: [];
    error_log('[age-check] ' . json_encode($row));
    return $row;
}

// -------------------------------------
// 9) CLI debug banner
// -------------------------------------
if (php_sapi_name() === 'cli') {
    fwrite(STDERR, "🔍 [TEST BOOTSTRAP] UTC now = {$GLOBALS['TEST_NOW_UTC']} | DB = " . TEST_DB_PATH . PHP_EOL);
}