# Orbit Unit Test Runner - PowerShell Wrapper
# ============================================
# 
# Comprehensive test runner for both Orbit-Admin and Orbit-Shop modules.
# This script runs unit tests for all available components in both directories.

# Clear console and set encoding
Clear-Host
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

# Change to script directory (project root)
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
Set-Location $ScriptDir

Write-Host "Working from: $(Get-Location)" -ForegroundColor Cyan
Write-Host "===============================================================" -ForegroundColor Yellow
Write-Host "Orbit Unit Test Suite - Running All Tests" -ForegroundColor Yellow
Write-Host "===============================================================" -ForegroundColor Yellow

$exitCode = 0

# Test Orbit-Shop
if (Test-Path "runTest.php") {
    Write-Host "`nRunning Orbit Tests..." -ForegroundColor Green
    Write-Host "-------------------------------------" -ForegroundColor Gray
    
    # Ensure reports directory exists
    if (!(Test-Path "reports")) {
        New-Item -ItemType Directory -Path "reports" -Force | Out-Null
    }
    
    # Generate timestamp for log file
    $orbitTimestamp = Get-Date -Format "yyyy-MM-dd_HH-mm-ss"
    $orbitLogFile = "reports/orbit-output_$orbitTimestamp.log"
    
    # Run tests and capture output
    php "runTest.php" 2>&1 | Tee-Object -FilePath $orbitLogFile
    
    if ($LASTEXITCODE -ne 0) {
        $exitCode = $LASTEXITCODE
        Write-Host "Tests failed with exit code: $LASTEXITCODE" -ForegroundColor Red
        Write-Host "Test log saved to: $orbitLogFile" -ForegroundColor Cyan
    } else {
        Write-Host "Orbit tests completed successfully" -ForegroundColor Green
        Write-Host "Test log saved to: $orbitLogFile" -ForegroundColor Cyan
    }
} else {
    Write-Host "Orbit test runner not found" -ForegroundColor Yellow
}

Write-Host "`n===============================================================" -ForegroundColor Yellow
if ($exitCode -eq 0) {
    Write-Host "All tests completed successfully!" -ForegroundColor Green
} else {
    Write-Host "Some tests failed. Check output above for details." -ForegroundColor Red
}
Write-Host "===============================================================" -ForegroundColor Yellow

# Pass through exit code
exit $exitCode