<?php

echo "🔄 Running Orbit-Core Post-Update Script...\n";

if (getenv('Test_ENV') === 'true') {
    echo "Test environment detected, skipping post-update script.\n";
    exit(0);
}

$rootDir = dirname(__DIR__, 4); // Go up four levels to reach the main project
$vendorDir = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'vendor' . DIRECTORY_SEPARATOR . 'adminprodigydesign' . DIRECTORY_SEPARATOR; // Vendor directory

$coreDir = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'src' . DIRECTORY_SEPARATOR;  // Where Orbit-Core files are stored
$coreSQLDir = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'SQL' . DIRECTORY_SEPARATOR; // SQL directory inside Orbit-Core
$coreFiles = ['db.php', 'Fcors.php', 'generic.php', 'error_monitor.php']; // Core files to move

// 1️⃣ Copy Orbit-Core files to the root directory
echo "🚀 Copying Orbit-Core files to root directory...\n";
foreach ($coreFiles as $file) {
    $source = $coreDir . $file;
    $destination = $rootDir . DIRECTORY_SEPARATOR . $file;

    if (file_exists($source)) {
        copy($source, $destination);
        echo "✅ Copied $file to project root.\n";
    } else {
        echo "⚠️ $file not found in Orbit-Core.\n";
    }
}

// 2️⃣ Debugging: List contents of $vendorDir to check if path is correct
echo "🔍 Listing contents of $vendorDir to verify path:\n";

$debugList = scandir($vendorDir);
if ($debugList === false) {
    echo "⚠️ Unable to read directory: $vendorDir\n";
} else {
    foreach ($debugList as $item) {
        echo " - $item\n";
    }
}

// 3️⃣ Detect and move other installed packages to `root/{package-name}/`
echo "🔍 Checking for other installed packages...\n";
$installedPackages = glob($vendorDir . '*', GLOB_ONLYDIR);

// Helper: recursively copy directory and always overwrite existing files
function copyDirectory(string $source, string $destination): void {
    if (!is_dir($destination)) {
        mkdir($destination, 0777, true);
    }

    $source = rtrim(str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $source), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;

    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($source, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($iterator as $file) {
        $filePath = str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $file->getPathname());
        $relativePath = substr($filePath, strlen($source));
        $targetPath = rtrim($destination, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $relativePath;

        if ($file->isDir()) {
            if (!is_dir($targetPath)) {
                mkdir($targetPath, 0777, true);
            }
        } else {
            $targetDir = dirname($targetPath);
            if (!is_dir($targetDir)) {
                mkdir($targetDir, 0777, true);
            }
            if (file_exists($targetPath)) {
                unlink($targetPath);
            }
            copy($file->getPathname(), $targetPath);
        }
    }
}

// Helper: recursively remove a directory and its contents
function removeDirectory(string $dir): void {
    if (!is_dir($dir)) {
        return;
    }

    $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );

    foreach ($it as $item) {
        if ($item->isDir()) {
            rmdir($item->getRealPath());
        } else {
            unlink($item->getRealPath());
        }
    }

    // Remove the top-level directory
    rmdir($dir);
}

foreach ($installedPackages as $packagePath) {
    $packageName = basename($packagePath);
    // temp stop so core is copyed for testing but added to git ignore so it does not cause issues for others
    //if ($packageName === 'orbit-core') {
    //    continue; // Skip Orbit-Core (already handled)
    //}

    // Extract package name from format `orbit-package`
    $packageNameParts = explode('-', $packageName, 2);
    if (count($packageNameParts) < 2) {
        echo "⚠️ Unable to determine package name for $packageName, skipping...\n";
        continue;
    }
    $packageName = $packageNameParts[1];

    $packageTarget = $rootDir . DIRECTORY_SEPARATOR . $packageName;
    if (!is_dir($packageTarget)) {
        mkdir($packageTarget, 0777, true);
    }

    echo "🚀 Moving files for $packageName to $packageTarget/...\n";
    
    // Move `src/` files
    $packageSrc = $packagePath . DIRECTORY_SEPARATOR . 'src' . DIRECTORY_SEPARATOR;
    if (is_dir($packageSrc)) {
        $files = glob($packageSrc . '*');
        foreach ($files as $file) {
            $filename = basename($file);
            $dest = $packageTarget . DIRECTORY_SEPARATOR . $filename;
            if (file_exists($dest)) {
                if (is_dir($dest)) {
                    removeDirectory($dest);
                } else {
                    unlink($dest);
                }
            }
            if (is_dir($file)) {
                // Recursively copy directories
                copyDirectory($file, $dest);
            } else {
                copy($file, $dest);
            }
        }
        echo "✅ Copied src/ files for $packageName.\n";
    } else {
        echo "⚠️ No src/ directory found in $packageName, skipping...\n";
    }

    // Move `SQL/` files
    $packageSQL = $packagePath . DIRECTORY_SEPARATOR . 'SQL' . DIRECTORY_SEPARATOR;
    if (is_dir($packageSQL)) {
        $sqlTarget = $packageTarget . DIRECTORY_SEPARATOR . 'SQL' . DIRECTORY_SEPARATOR;
        if (!is_dir($sqlTarget)) {
            mkdir($sqlTarget, 0777, true);
        }
        
        $files = glob($packageSQL . '*');
        foreach ($files as $file) {
            $filename = basename($file);
            $dest = $sqlTarget . $filename;
            if (file_exists($dest)) {
                if (is_dir($dest)) {
                    removeDirectory($dest);
                } else {
                    unlink($dest);
                }
            }
            if (is_dir($file)) {
                copyDirectory($file, $dest);
            } else {
                copy($file, $dest);
            }
        }
        echo "✅ Copied SQL/ files for $packageName.\n";
    } else {
        echo "⚠️ No SQL/ directory found in $packageName, skipping...\n";
    }

    // Move `tests/` files
    $packageTests = $packagePath . DIRECTORY_SEPARATOR . 'tests' . DIRECTORY_SEPARATOR;
    if (is_dir($packageTests)) {
        $testsTarget = $packageTarget . DIRECTORY_SEPARATOR . 'tests' . DIRECTORY_SEPARATOR;
        if (!is_dir($testsTarget)) {
            mkdir($testsTarget, 0777, true);
        }
        
        $files = glob($packageTests . '*');
        foreach ($files as $file) {
            $filename = basename($file);
            $dest = $testsTarget . $filename;
            if (file_exists($dest)) {
                if (is_dir($dest)) {
                    removeDirectory($dest);
                } else {
                    unlink($dest);
                }
            }
            if (is_dir($file)) {
                copyDirectory($file, $dest);
            } else {
                copy($file, $dest);
            }
        }
        echo "✅ Copied tests/ files for $packageName.\n";
    } else {
        echo "⚠️ No tests/ directory found in $packageName, skipping...\n";
    }

    // Special handling for core package - move `unit-test/` directory
    if ($packageName === 'core') {
        $packageUnitTest = $packagePath . DIRECTORY_SEPARATOR . 'unit-test' . DIRECTORY_SEPARATOR;
        if (is_dir($packageUnitTest)) {
            $unitTestTarget = $packageTarget . DIRECTORY_SEPARATOR . 'unit-test' . DIRECTORY_SEPARATOR;
            
            
            
            copyDirectory($packageUnitTest, $unitTestTarget);
            echo "✅ Copied unit-test/ directory for core.\n";
        } else {
            echo "⚠️ No unit-test/ directory found in core, skipping...\n";
        }

        // Move `misc/` directory for core package
        $packageMisc = $packagePath . DIRECTORY_SEPARATOR . 'misc' . DIRECTORY_SEPARATOR;
        if (is_dir($packageMisc)) {
            $miscTarget = $packageTarget . DIRECTORY_SEPARATOR . 'misc' . DIRECTORY_SEPARATOR;
            copyDirectory($packageMisc, $miscTarget);
            echo "✅ Copied misc/ directory for core.\n";
        } else {
            echo "⚠️ No misc/ directory found in core, skipping...\n";
        }
    }
}

echo "✅ Post-Update Process Complete!\n";

// Attempt to regenerate phpunit.xml so pushed changes update test suites
$generator = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'unit-test' . DIRECTORY_SEPARATOR . 'generate-phpunit.php';
if (file_exists($generator)) {
    echo "🔧 Generating phpunit.xml using generator: $generator\n";
    $exclude = getenv('TEST_EXCLUDE') ?: '';
    $cmd = escapeshellarg(PHP_BINARY) . ' ' . escapeshellarg($generator) . ($exclude !== '' ? ' --exclude=' . escapeshellarg($exclude) : '');
    passthru($cmd, $exitCode);
    if ($exitCode !== 0) {
        echo "⚠️ phpunit generator exited with code $exitCode\n";
    } else {
        echo "✅ phpunit.xml generated successfully.\n";
    }
} else {
    echo "⚠️ phpunit generator not found at $generator, skipping generation.\n";
}

?>
