#!/bin/bash

# Orbit Unit Test Runner - Bash Version
# =====================================

# Change to project root directory
cd "$(dirname "$0")/../.."
echo -e "\033[36m📁 Working from: $(pwd)\033[0m"

clear

# Force UTF-8 encoding and enable error display
export LC_ALL=C.UTF-8
export LANG=C.UTF-8
export PHPUNIT_DISPLAY_ERRORS=1

# Function to create display name from module name
create_display_name() {
    local module="$1"
    case "$module" in
        "core") echo "Core Tests" ;;
        "cms") echo "cms Tests" ;;
        "cms-image") echo "cms Images Tests" ;;
        "login") echo "Login Tests" ;;
        "tenant") echo "Tenant Tests" ;;
        "meta-data") echo "Meta Data Tests" ;;
        "food-menu") echo "Food Menu Tests" ;;
        "food-booking") echo "Food Booking Tests" ;;
        "food-order-public") echo "Food Order Public Tests" ;;
        "food-order-tenant") echo "Food Order Tenant Tests" ;;
        "food-order-super-admin") echo "Food Order Super Admin Tests" ;;
        "wood-product") echo "Wood Product Tests" ;;
        *)
            # Convert kebab-case to Title Case
            echo "$module" | sed 's/-/ /g' | awk '{for(i=1;i<=NF;i++) $i=toupper(substr($i,1,1)) tolower(substr($i,2))} 1' | sed 's/$/ Tests/'
            ;;
    esac
}

# Auto-update phpunit.xml testsuites section
echo -e "\033[33m🔄 Updating phpunit.xml testsuites...\033[0m"

# Define excluded folders
excluded_folders=("vendor" "tests" "scripts" "Default Example" ".git" ".phpunit.cache")

# Define dependency order for modules
module_order=("core" "cms" "cms-image" "login" "tenant" "food-menu" "food-booking" "food-order-public" "food-order-tenant" "food-order-super-admin" "wood-product" "meta-data")

# Find all valid modules
declare -a all_modules
while IFS= read -r -d '' dir; do
    module_name=$(basename "$dir")
    
    # Check if module should be excluded
    excluded=false
    for exclude in "${excluded_folders[@]}"; do
        if [[ "$module_name" == "$exclude" ]]; then
            excluded=true
            break
        fi
    done
    
    # Check if module has tests and composer.json
    if [[ "$excluded" == false ]] && [[ -d "$dir/tests" ]] && [[ -f "$dir/composer.json" ]]; then
        all_modules+=("$module_name")
    fi
done < <(find . -maxdepth 1 -type d -print0)

# Sort modules by dependency order first, then alphabetically
declare -a sorted_modules

# Add modules in predefined order
for ordered_module in "${module_order[@]}"; do
    for module in "${all_modules[@]}"; do
        if [[ "$module" == "$ordered_module" ]]; then
            sorted_modules+=("$module")
            break
        fi
    done
done

# Add any remaining modules not in predefined order
for module in "${all_modules[@]}"; do
    found=false
    for ordered_module in "${module_order[@]}"; do
        if [[ "$module" == "$ordered_module" ]]; then
            found=true
            break
        fi
    done
    if [[ "$found" == false ]]; then
        sorted_modules+=("$module")
    fi
done

# Sort remaining modules alphabetically
IFS=$'\n' sorted_remaining=($(sort <<<"${sorted_modules[*]:${#module_order[@]}}"))
sorted_modules=("${sorted_modules[@]:0:${#module_order[@]}}" "${sorted_remaining[@]}")

if [[ ${#sorted_modules[@]} -gt 0 ]]; then
    # Generate testsuite XML
    testsuite_xml="    <testsuites>\n"
    for module in "${sorted_modules[@]}"; do
        display_name=$(create_display_name "$module")
        testsuite_xml+="        <testsuite name=\"$display_name\">\n"
        testsuite_xml+="            <directory>$module/tests</directory>\n"
        testsuite_xml+="        </testsuite>\n"
    done
    testsuite_xml+="    </testsuites>"
    
    # Update phpunit.xml
    if [[ -f "../../core/unit-test/phpunit.xml" ]]; then
        # Create backup
        cp ../../core/unit-test/phpunit.xml ../../core/unit-test/phpunit.xml.backup
        
        # Use perl for multi-line regex replacement
        perl -0pe 's|<testsuites>.*?</testsuites>|'"$(echo -e "$testsuite_xml")'|s' ../../core/unit-test/phpunit.xml > ../../core/unit-test/phpunit.xml.tmp && mv ../../core/unit-test/phpunit.xml.tmp ../../core/unit-test/phpunit.xml
        
        echo -e "\033[32m✅ Updated phpunit.xml with ${#sorted_modules[@]} test modules\033[0m"
    else
        echo -e "\033[33m⚠️ phpunit.xml not found\033[0m"
    fi
else
    echo -e "\033[33m⚠️ No test modules found\033[0m"
fi

# Set TEST_ENV so the PHP script skips moving files
export TEST_ENV=true

# Backup the existing composer files
echo -e "\033[33m💾 Backing up composer files...\033[0m"
cp composer.json composer.json.bak 2>/dev/null || echo "No composer.json to backup"
cp composer.lock composer.lock.bak 2>/dev/null || echo "No composer.lock to backup"

# Use the test composer.json by copying it
cp composer.test.json composer.json
cp composer.test.lock composer.lock 2>/dev/null || echo "No composer.test.lock found"

# Remove vendor and lock files to ensure fresh installation
echo -e "\033[33m🧹 Cleaning vendor directory...\033[0m"
rm -rf vendor/adminprodigydesign 2>/dev/null
rm -f composer.lock 2>/dev/null

# Clear Composer cache and install dependencies
echo -e "\033[33m📦 Installing dependencies...\033[0m"
composer dump-autoload
composer clear-cache
composer update --no-scripts --ignore-platform-reqs

# Pause for 5 seconds
echo -e "\033[33mPausing for 5 seconds...\033[0m"
sleep 5

clear

# Generate timestamp for logs
timestamp=$(date '+%Y-%m-%d_%H-%M-%S')
timestamp2=$(date '+%Y-%m-%d %H:%M:%S')

# Define log file paths
testdox_log="core/unit-test/reports/testdox/testdox-log_$timestamp.txt"
junit_log="core/unit-test/reports/junit/junit-log_$timestamp.xml"
warnings_log="core/unit-test/reports/warning/warnings-log_$timestamp.txt"
full_log="core/unit-test/reports/full-output_$timestamp.log"

# Ensure reports directories exist
mkdir -p core/unit-test/reports/testdox
mkdir -p core/unit-test/reports/junit
mkdir -p core/unit-test/reports/warning

# Run PHPUnit tests
echo "Running tests..."
echo -e "\033[34mTest started at $timestamp2\033[0m"

# Verify PHPUnit installation
if [[ -f "vendor/bin/phpunit" ]]; then
    # Parse testsuites from phpunit.xml and run each individually
    testsuites=($(grep -o '<testsuite name="[^"]*"' ../../core/unit-test/phpunit.xml | sed 's/<testsuite name="//g' | sed 's/"//g'))
    
    echo -e "\033[36mRunning ${#testsuites[@]} test suites individually...\033[0m"
    
    overall_exit_code=0
    
    for testsuite in "${testsuites[@]}"; do
        echo -e "\n\033[33m🧪 Running: $testsuite\033[0m"
        
        # Run the specific testsuite
        vendor/bin/phpunit --configuration "../../core/unit-test/phpunit.xml" --testsuite "$testsuite" --testdox --log-junit "$junit_log" --testdox-text "$testdox_log" --display-warnings 2> "$warnings_log" | tee -a "$full_log"
        
        exit_code=${PIPESTATUS[0]}
        
        if [[ $exit_code -ne 0 ]]; then
            echo -e "\033[31m❌ $testsuite failed with exit code $exit_code\033[0m"
            overall_exit_code=$exit_code
        else
            echo -e "\033[32m✅ $testsuite completed successfully\033[0m"
        fi
    done
    
    # Cleanup: remove temporary SQLite file
    php core/unit-test/scripts/cleanup.php 2>/dev/null || true
    
else
    echo -e "\033[31m❌ PHPUnit is not installed. Check composer.test.json and try again.\033[0m"
    exit 1
fi

# Display Summary
echo ""
echo -e "\033[34mTest completed at $(date '+%Y-%m-%d %H:%M:%S')\033[0m"
echo "Full Log: $full_log"
echo "Warnings Log: $warnings_log"
echo "TestDox Log: $testdox_log"

# Restore composer files
echo -e "\033[33m🔄 Restoring composer configuration...\033[0m"
mv composer.json composer.test.json
# mv composer.lock composer.test.lock 2>/dev/null || true

# Restore original composer files
mv composer.json.bak composer.json
mv composer.lock.bak composer.lock

# Reset TEST_ENV
unset TEST_ENV

echo -e "\033[32m🎉 Test run completed!\033[0m"

exit $overall_exit_code